<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class Request extends Model
{
    
    use SoftDeletes;

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'canceled_at', 'created_at', 'updated_at', 'released_at'
    ];

    /**
     * Loads user related to this request
     * @return void
     */
    public function user() {
        return $this->belongsTo(User::class);
    }

    /**
     * Loads partnership related to this Request
     * @return void 
     */
    public function businessClient() {
        return $this->belongsTo(BusinessClient::class);
    }

    /**
     * Loads deposits related to Request
     */
    public function deposits() {
        return $this->hasMany(RequestDeposit::class);
    }

    /**
     * Loads files to request
     * @return mixed 
     */
    public function paymentFiles() {
        return $this->hasMany(RequestFile::class)->where('type', 'PAY');
    }

    /**
     * Loads employee scatter to request
     * @return HasMany 
     */
    public function employees() {
        return $this->hasMany(EmployeeScatter::class);
    }

    /**
     * Loads employee scatter to request
     * @return HasMany 
     */
    public function employeesWithEmployee() {
        return $this->hasMany(EmployeeScatter::class)->with('employee');
    }

    /**
     * Loads payroll workers
     */
    public function employeesWithFiles() {
        return $this->hasMany(EmployeeScatter::class)->with('employee', 'scatterDetails');
    }

    public function employeesWithAll() {
        return $this->hasMany(EmployeeScatter::class)->with('employee', 'payrollWorkerReceipts', 'scatterDetails');
    }

    public function getEmployee($id) {
        $employees = $this->employeesWithAll;
        foreach ($employees as $employee) {
            if ($employee->employee_id == $id) {
                return $employee;
            }
        }
        return null;
    }

    /**
     * Loads invoices related to request
     * @return HasMany 
     */
    public function requestInvoices() {
        return $this->hasMany(RequestInvoice::class);
    }
    
    public function invoices() {
        return $this->hasMany(Invoice::class);
    }

    /**
     * Loads invoices related to request
     * @return HasMany 
     */
    public function requestFilesDeposits() {
        return $this->hasMany(RequestFile::class)->where('type', 'receipt-pay');
    }

    /**
     * Loads branches related to client
     */
    public function branches() {
        return $this->hasMany(BusinessBranch::class);
    }

    /**
     * Loads payroll workers
     */
    public function payrollWorkers() {
        return $this->hasMany(PayrollWorker::class);
    }

    /**
     * Loads payroll worker
     */
    public function getPayrollWorker() {
        if (count($this->payrollWorkers) > 0) {
            return $this->payrollWorkers[0];
        }
        return null;
    }

    /**
     * Gets a scatter file name
     * @return mixed 
     */
    public function getScatterFile() {
        if ($this->scatter_file) {
            return substr($this->scatter_file, 
                strlen($this->businessClient->code) + 4 + strlen($this->id));
        }
        return '';
    }

    public function getRequested() {
        $amount = 0;
        foreach ($this->employees as $employee) {
            $amount += $employee->amount;
        }
        return $amount;
    }

    public function getStatus() {
        if ($this->status == 'O') {
            return 'Abierto';
        } else if ($this->status == 'P') {
            return 'Pendiente';
        } else if ($this->status == 'R') {
            return 'Liberado';
        } else if ($this->status == 'F') {
            return 'Rechazado';
        }
        return '';
    }

    public function isReleased() {
        return $this->status == 'R';
    }

    public function HasStampReceipts() {
        if (count($this->payrollWorkers) > 0) {
            $payroll = $this->payrollWorkers[0];
            return PayrollWorkerReceipt::where('payroll_worker_id', $payroll->id)
                ->whereNotNull('uuid')
                ->count() > 0;
        }        
        return false;
    }

    public function canSendRequest() {
        if ($this->status == 'F') {
            return false;
        }
        $employees = $this->employeesWithAll;
        foreach ($employees as $employeeScatter) {
            if ($employeeScatter->getPendingAmount() > 0) {
                return true;
            }
        }
        return $this->id;// && count($this->deposits) > 0;
    }

    public function isRequestComplete() {
        return $this->HasStampReceipts();
    }

    public function getTotalScatter() {
        $amount = 0;
        $employees =  $this->employeesWithAll;
        foreach ($employees as $employee) {
            $amount += $employee->amount + $employee->getPendingAmount();
        }
        return $amount;
    }

    public function hasEmployeeScatterSending() {
        $employees = $this->employeesWithFiles;
        foreach ($employees as $employee) {
            foreach ($employee->scatterDetails as $detail) {
                if ($detail->isSending()) {
                    if (Auth::user()->type == 'BI' && !$detail->fac_status) {
                        return true;
                    } else if (Auth::user()->type == 'BA' && !$detail->bank_status) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public function hasEmployeeFiles() {
        $employees = $this->employeesWithFiles;
        foreach ($employees as $employee) {
            if ($employee->hasEmployeeFiles()) {
                return true;
            }
        }
        return false;
    }
}
