<?php

namespace App\Model;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PayrollWorkerReceipt extends Model
{
    use SoftDeletes;

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'started_at', 'invoice_date', 'pay_date', 'pay_init', 'pay_end', 'stamped_at', 'canceled_at', 'deleted_at', 
        'created_at', 'updated_at'
    ];

    /**
     * Loads Payroll header
     */
    public function payrollWorker() {
        return $this->belongsTo(PayrollWorker::class);
    }

    public function employeeScatter() {
        return $this->belongsTo(EmployeeScatter::class);
    }

    public function employee() {
        return $this->belongsTo(Employee::class);
    }

    public function workerReceiptReductions() {
        return $this->hasMany(WorkerReceiptReduction::class);
    }

    public function getSalaryDayAttribute() {
        if ($this->integration_factor > 0) {
            return round($this->sdi / $this->integration_factor, 2);
        }
        return 0;
    }

    public function getSalaryTotalAttribute() {
        return round($this->salary_day * $this->work_days, 2);
    }

    public function getRemainingSalaryAttribute() {
        return $this->employeeScatter->amount - ($this->total);
    }

    public function isStamped() {
        return $this->uuid;
    }

    public function getFullNameAttribute() {
        $name = trim($this->first_name);
        if ($this->middle_name) {
            $name .= ' ' . trim($this->middle_name);
        }
        if ($this->last_name) {
            $name .= ' ' . trim($this->last_name);
        }
        return $name;
    }

    public function getPeriodLegendAttribute() {
        $code = $this->payment_period;
        if ($code === '01') {
            return '01 Diario';
        }
        if ($code === '02') {
            return '02 Semanal';
        }
        if ($code === '03') {
            return '03 Catorcenal';
        }
        if ($code === '04') {
            return '04 Quincenal';
        }
        if ($code === '05') {
            return '05 Mensual';
        }
        if ($code === '06') {
            return '06 Bimestral';
        }
        if ($code === '07') {
            return '07 Unidad obra';
        }
        if ($code === '08') {
            return '08 Comisión';
        }
        if ($code === '09') {
            return '09 Precio alzado';
        }
        if ($code === '10') {
            return '10 Decenal';
        }
        if ($code === '99') {
            return '99 Otra Periodicidad';
        }
        return '';
    }

    public function getStampPath() {
        if ($this->stamped_at) {
            $businessClient = $this->employee->businessClient;
            return $businessClient->code . '/' . $this->stamped_at->format('Ymd') . '/';
        }
        return '';
    }

    public function getSysAttribute() {
        return $this->total;
    }

    public function getAsAttribute() {
        return $this->remaining_salary;
    }

    public function getAntiquityAttribute() {
        $from = Carbon::createFromFormat('d-m-Y', $this->started_at->format('d-m-Y')); 
        $to = Carbon::createFromFormat('d-m-Y', $this->pay_end->format('d-m-Y')); 

        $days = $from->addDays(-1)->diffInDays($to);
        $ant = "P";
        $weeks = intdiv($days, 7);
        if ($weeks > 0) {
            $ant .= $weeks . "W";
        }        
        return $ant;
    }

    public function getReductionsAttribute() {
        $total = 0;
        $reductions = $this->workerReceiptReductions;
        foreach ($reductions as $reduction) {
            $total += $reduction->amount;
        }
        return $total;
    }
}
