<?php

namespace App\Model;

use App\InvoicePayment;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    
    use SoftDeletes;

    /**
     * The attributes that should be mutated to dates
     */
    protected $dates = [
        'stamped_at', 'invoice_at', 'created_at', 'updated_at', 'deleted_at'
    ];

    /**
     * Retrieves business client emissor of invoice
     */
    public function emissor() {
        return $this->belongsTo(BusinessClient::class, 'emissor_id');
    }

    /**
     * Retrieves business client receiver of invoice
     */
    public function receiver() {
        return $this->belongsTo(BusinessClient::class, 'receiver_id');
    }

    /**
     * Retrieves branch of business
     */
    public function businessBranch() {
        return $this->belongsTo(BusinessBranch::class);
    }

    /**
     * Retrieves pay type of invoice
     */
    public function invoicePayType() {
        return $this->belongsTo(InvoicePayType::class);
    }

    /**
     * Retrieves taxregime of invoice
     */
    public function invoiceTaxRegime() {
        return $this->belongsTo(InvoiceTaxRegimes::class);
    }

    /**
     * Retrieves type of invoice
     */
    public function invoiceType() {
        return $this->belongsTo(InvoiceType::class);
    }

    /**
     * Retrieves use of invoice
     */
    public function invoiceUse() {
        return $this->belongsTo(InvoiceUse::class);
    }

    /**
     * Retrieves concepts of invoice
     */
    public function invoiceConcepts() {
        return $this->hasMany(InvoiceConcept::class);
    }

    /**
     * Retrieves payment linked to Invoice
     */
    public function payments() {
        return $this->hasMany(InvoicePayment::class, 'invoice_id');
    }

    public function invoicePayments() {
        return $this->hasMany(InvoicePayment::class, 'invoice_related_id');
    }

    /**
     * Get pay method description
     * 
     * PUE - Pago en una sola exhibición
     * PPD - Pago en parcialidades o diferido
     */
    public function getPayMethodDescription() {
        if ($this->pay_method === 'PUE') {
            return 'Pago en una sola exhibición';
        } else if ($this->pay_method === 'PPD') {
            return 'Pago en parcialidades o diferido';
        }
        return '';
    }

    /**
     * Calc total of concepts
     */
    public function getTotalConcepts() {
        $concepts = $this->invoiceConcepts;
        $carry = 0;
        foreach ($concepts as $concept) {
            $carry += $concept->total;
        }
        return $carry;
    }

    /**
     * remove retention to total invoice
     * @return total invoice
     */
    public function calcTotalNeto() {
        $total = $this->total;
        if ($this->receiver != Null && $this->receiver->applyRetention()) {
            $total -= $this->retention;
        }
        return $total;
    }

    /**
     * Retrives a concept by id
     */
    public function getInvoiceConcept($id) {
        $concepts = $this->invoiceConcepts;
        foreach ($concepts as $concept) {
            if ($concept->id == $id) {
                return $concept;
            }
        }
        return null;
    }

    /**
     * Gets if can stamp the invoice
     * 
     */
    public function canStamp() {
        return !!$this->id;
    }

    /**
     * Gets if is stamped the invoice
     */
    public function isStamped() {
        return $this->uuid;
    }
    
    /**
     * Gets the path where has files stampeds
     */
    public function getStampPath() {
        $businessClient = $this->emissor;
        return $businessClient->code . '/' . $this->stamped_at->format('Ymd') . '/';
    }
    
    public function getCarriedAmount() {
        $amount = 0;
        foreach ($this->invoicePayments as $payment) {
            if ($payment->invoice->isStamped() && !$payment->invoice->canceled_at) {
                $amount = $payment->carried_amount;
            }
        }
        return $amount;
    }

    public function getAmountPayed() {
        $amount = 0;
        foreach ($this->invoicePayments as $payment) {
            if ($payment->invoice->isStamped() && !$payment->invoice->canceled_at) {
                $amount += $payment->amount;
            }
        }
        return $amount;
    }
}
