<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;

class BusinessClient extends Model
{
    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'released_at', 'canceled_at', 'created_at', 'updated_at'
    ];
    
    /**
     * Loads branches related to client
     */
    public function branches() {
        return $this->hasMany(BusinessBranch::class);
    }

    /**
     * Loads departments related to client
     */
    public function departments() {
        return $this->hasMany(BusinessDepartment::class);
    }

    /**
     * Loads banks related to client
     * @return void 
     */
    public function banks() {
        return $this->hasMany(BusinessClientBank::class);
    }

    /**
     * Loads partnership related to client
     * @return Partnership related
     */
    public function partnership() {
        return $this->belongsTo(Partnership::class);
    }

    /**
     * Loads user that released
     */
    public function releasedBy() {
        return $this->belongsTo(User::class, 'released_by');
    }

    /**
     * Loads a department related to client
     * 
     * @param code The code from department
     */
    public function getDepartment($code) {
        foreach ($this->departments as $department) {
            if ($department->code === $code) {
                return $department;
            }
        }
        return null;
    }
    /**
     * gets a pattern for codes
     */
    public function getCodePattern($branchId) {
        foreach ($this->branches as $b) {
            if ($b->id == intval($branchId)) {
                return $this->prefix . '-' . $b->prefix . '-';
            }
        }
        return $this->prefix . '-';
    }

    /**
     * Gets a code type for build a code for stamp
     */
    public function getTypeForCodes() {
        return 'I-' . $this->code;
    }

    /**
     * Valids if the current clietn has configurated files
     */
    public function hasFilesForStamp() {
        if (!$this->key_file || trim($this->key_file) == '') {
            return false;
        }
        if (!$this->password_file || trim($this->password_file) == '') {
            return false;
        }
        if (!$this->certificate_file || trim($this->certificate_file) == '') {
            return false;
        }
        return true;
    }

    /**
     * Indicates if the client apply retentions
     * @return void 
     */
    public function applyRetention() {
        return $this->retention != NULL && $this->retention > 0;
    }

    /**
     * Gets a rfc file name
     * @return mixed 
     */
    public function getRFCFile() {
        if ($this->rfc_file) {
            $dir = explode("/", $this->rfc_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Gets a address file name
     * @return mixed 
     */
    public function getAddressFile() {
        if ($this->address_file) {
            $dir = explode("/", $this->address_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Gets a constitutive act file name
     * @return mixed 
     */
    public function getConstitutiveActFile() {
        if ($this->constitutive_act_file) {
            $dir = explode("/", $this->constitutive_act_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Gets a ife file name
     * @return mixed 
     */
    public function getIFEFile() {
        if ($this->ife_file) {
            $dir = explode("/", $this->ife_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Gets a identification card file name
     * @return mixed 
     */
    public function getIdentificationCardFile() {
        if ($this->identification_card_file) {
            $dir = explode("/", $this->identification_card_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Gets a main identification card file name
     * @return mixed 
     */
    public function getMainIdentificationCardFile() {
        if ($this->main_identification_card_file) {
            $dir = explode("/", $this->main_identification_card_file);
            return $dir[count($dir) - 1];
        }
        return '';
    }

    /**
     * Format user data
     */
    public function format() {
        return [
            'id' => $this->id,
            'code' => $this->code,
            'name' => $this->name,
            'rfc' => $this->rfc,
            'prefix' => $this->prefix,
            'certificateFile' => $this->certificate_file,
            'keyFile' => $this->key_file,
            'passwordFile' => $this->password_file
        ];
    }

    public function getStatus() {
        if ($this->status == 'O') {
            return 'Abierto';
        } else if ($this->status == 'P') {
            return 'Pendiente';
        } else if ($this->status == 'R') {
            return 'Liberado';
        } else if ($this->status == 'F') {
            return 'Rechazado';
        }
        return '';
    }
    
    public function getType() {
        if ($this->type == 'M') {
            return 'Persona moral';
        }
        return 'Persona fisica';
    }
}
