<?php

namespace App\Http\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use phpseclib\File\X509;

/**
 * Allow invoice cfdi data
 */
class InvoiceService {

    /**
     * Stamped string
     * @param xml Certificated XML
     */
    public static function getOriginalString($xml): string {
        $xsl = new \DOMDocument();
        $url = Storage::disk('xslt')->getDriver()->getAdapter()
            ->getPathPrefix();
        $xsl->load($url . 'cadenaoriginal_3_3.xslt');

        $xmlDocument = new \DOMDocument();
        $xmlDocument->loadXML($xml);

        $xslt = new \XSLTProcessor();
        $xslt->importStylesheet($xsl);
        return $xslt->transformToXML($xmlDocument);
    }

    public static function stamp($xml, $key, $password): string {
        Log::info('Stamping');
        $xmlData = simplexml_load_string($xml);
        $originalString = InvoiceService::getOriginalString($xml);

        $url = Storage::disk('cert')->getDriver()->getAdapter()
            ->getPathPrefix() . $key;
        $pwd = trim(Storage::disk('cert')->get($password));

        $command = 'openssl pkcs8 -in ' .
            $url . ' -inform DER -passin pass:' . $pwd;

        $output = [];
        exec ($command, $output, $response);
        $private = join("\n", $output);

        $sig = '';
        openssl_sign($originalString, $sig, $private, OPENSSL_ALGO_SHA256);
        $stamp = base64_encode($sig);

        $xmlData->attributes()['Sello'] = $stamp;
        return $xmlData->asXML();
    }

    /**
     * Invoice process
     */
    public static function certificate($xml, $certFile): string {
        Log::info('Certificating');
        $xmlData = simplexml_load_string($xml);

        $cert = base64_encode(Storage::disk('cert')->get($certFile));
        $xmlData->attributes()['Certificado'] = $cert;
        $xmlData->attributes()['NoCertificado'] = InvoiceService::getCertNumber($certFile);
        return $xmlData->asXML();
    }

    /**
     * Retrieve certificate number
     */
    private static function getCertNumber($certFile): string {
        $x509 = new X509();
        $data = $x509->loadX509(Storage::disk('cert')->get($certFile));
        $decString = (string)$data['tbsCertificate']['serialNumber'];

        if (0 === strpos($decString, '0x') || 0 === strpos($decString, '0X')) {
            $hexString = substr($decString, 2);
        } else {
            $hexString = BaseConverter::createBase36()
                ->convert($decString, 10, 16);
        }

        $certNumber = array_reduce(str_split($hexString, 2),
            function (string $carry, string $value): string {
                return $carry . chr(intval(hexdec($value)));
            }, '');
        return $certNumber;
    }



}
