<?php

namespace App\Http\Services;

use Carbon\Carbon;
use DateTime;
use Exception;
use DOMDocument;
use Illuminate\Support\Facades\Log;
use XSLTProcessor;
use Illuminate\Support\Facades\Storage;

/**
 * Provide document services for load and transform
 */
class DocumentServices {

    /**
     * Transform XSLT and Object data (XML) into CFDI xml
     */
    public static function buildCFDI(Array $data, $documentXSL)
    {
        try {
            $xml = new DOMDocument();
            $xml->loadXML(DocumentServices::arrayToXml($data));
            $xsl = new DOMDocument;
            $xsl->loadXML(Storage::disk('local')->get($documentXSL));

            $proc = new XSLTProcessor();
            $proc->importStyleSheet($xsl);

            return $proc->transformToXML($xml);
        } catch(Exception $ex) {
            report($ex);
            throw new Exception('Error to build CFDI', -1, $ex);
        }
    }

    /**
     * Transform XSLT and Object data (XML) into CFDI xml
     */
    public static function buildAsalariedCFDI(Array $data)
    {
        try {
            $xml = new DOMDocument();
            $xml->loadXML(DocumentServices::arrayToXml($data));

            $xsl = new DOMDocument;
            $xsl->loadXML(Storage::disk('local')->get('documents/AsalariedCFDI.xsl'));

            $proc = new XSLTProcessor();
            $proc->importStyleSheet($xsl);

            return $proc->transformToXML($xml);
        } catch(Exception $ex) {
            report($ex);
            throw new Exception('Error to build CFDI', -1, $ex);
        }
    }

    /**
     * Create XML from assoc array
     */
    private static function arrayToXml($array, $rootElement = null, $xml = null) {
        $_xml = $xml;

        // If there is no Root Element then insert root
        if ($_xml === null) {
            $_xml = new \SimpleXMLElement($rootElement !== null ? $rootElement : '<data/>');
        }

        // Visit all key value pair
        foreach ($array as $k => $v) {
            // If there is nested array then
            if (is_array($v)) {
                // Call function for nested array
                if (is_numeric($k)) {
                    DocumentServices::arrayToXml($v, $k, $_xml);
                } else {
                    DocumentServices::arrayToXml($v, $k, $_xml->addChild( \htmlspecialchars($k)));
                }
            } else {
                // Simply add child element.
                $_xml->addChild($k, \htmlspecialchars($v));
            }
        }

        return $_xml->asXML();
    }

    public static function buildXMLContent($payrollReceipt, $code = null) {
        $tipoRegimen = '11';
        $tipoNomina = 'O';
        if (!$code) {
            $code = 'TEMP';
        }
        $data = array(
            "Folio" => $code,
            "Fecha" => $payrollReceipt->invoice_date->format('Y-m-d\TH:i:s'),
            "Emisor" => array (
                "Rfc" => $payrollReceipt->payroll->rfc,
                "Nombre" => $payrollReceipt->payroll->name
            ),
            "Receptor" => array(
                "Rfc" => $payrollReceipt->rfc,
                "Nombre" => $payrollReceipt->getFullName(),
                "ClaveEntFed" => $payrollReceipt->clave_ent_fed,
                "NumEmpleado" => $payrollReceipt->num_empleado,
                "TipoRegimen" => $tipoRegimen,
                "curp" => $payrollReceipt->curp
            ),
            "Conceptos" => [
                array(
                    "Concepto" => array (
                        "ClaveProdServ" => "84111505"
                ))
            ],
            "nomina" => array(
                "dias" => $payrollReceipt->work_days,
                "FechaPago" => $payrollReceipt->pay_date->format('Y-m-d'),
                "FechaInicialPago" => $payrollReceipt->pay_init->format('Y-m-d'),
                "FechaFinalPago" => $payrollReceipt->pay_end->format('Y-m-d'),
                "periodicidad" => $payrollReceipt->pay_period,
                "tipoNomina" => $tipoNomina,
                "subtotal" => $payrollReceipt->subtotal,
                "isr" => $payrollReceipt->isr,
                "total" => $payrollReceipt->total
            )
        );
        return DocumentServices::getCFDI($data, 'documents/cfdi.xsl');
    }

    public static function buildSysXMLContent($receipt, $code = null) {
        if (!$code) {
            $code = 'TEMP';
        }
        $subtotal = $receipt->subsidy + $receipt->salary_total;
        $payroll = $receipt->payrollWorker;
        
        $ant = $receipt->antiquity;
        //$receipt->employee->federative_key
        $branch = $receipt->employee->getBusinessBranch();

        $deducciones = [];
        if ($receipt->isr > 0) {
            array_push($deducciones, [[
                "Deduccion" => [
                    "TipoDeduccion" => "002",
                    "Clave" => "002",
                    "Concepto" => "Pago de nómina",
                    "Importe" => round($receipt->isr, 2)
            ]]]);
        }
        if ($receipt->infonavit > 0) {
            array_push($deducciones, [[
                "Deduccion" => [
                    "TipoDeduccion" => "010",
                    "Clave" => "010",
                    "Concepto" => "Pago de nómina",
                    "Importe" => round($receipt->infonavit, 2)
            ]]]);
        }
        if ($receipt->imss > 0) {
            array_push($deducciones, [[
                "Deduccion" => [
                    "TipoDeduccion" => "021",
                    "Clave" => "021",
                    "Concepto" => "Pago de nómina",
                    "Importe" => round($receipt->imss, 2)
            ]]]);
        }

        $otherReductions = $receipt->workerReceiptReductions;
        foreach ($otherReductions as $reduction) {
            array_push($deducciones, [[
                "Deduccion" => [
                    "TipoDeduccion" => $reduction->code,
                    "Clave" => $reduction->code,
                    "Concepto" => $reduction->concept,
                    "Importe" => round($reduction->amount, 2)
            ]]]);
        }


        $descuento = $receipt->imss + $receipt->infonavit + $receipt->isr + $receipt->reductions;
        $data = array(
            "Folio" => $code,
            "Fecha" => $payroll->invoiced_at->format('Y-m-d\TH:i:s'),
            "FormaPago" => "99",
            "Subtotal" => round($subtotal, 2),
            "Descuento" => round($descuento, 2),
            "Total" => round(round($subtotal, 2) - round($descuento, 2), 2),
            "LugarExpedicion" => $branch->postal_code,
            "Emisor" => array (
                "Rfc" => $receipt->employee->businessClient->rfc,
                "Nombre" => $receipt->employee->businessClient->name,
                "RegimenFiscal" => "601"
            ),
            "Receptor" => array(
                "Rfc" => $receipt->rfc,
                "Nombre" => $receipt->full_name,
                "UsoCFDI" => "P01",
                "curp" => $receipt->curp,
            ),
            "Conceptos" => [
                array(
                    "Concepto" => array (
                        "ClaveProdServ" => "84111505",
                        "valorUnitario" => round($subtotal, 2),
                        "importe" => round($subtotal, 2),
                        "descuento" => round($descuento, 2)
                ))
            ],
            "Deducciones" => $deducciones,
            "nomina" => array(
                "registroPatronal" => $receipt->employee->businessClient->employer_registration,
                "tipoNomina" => $receipt->employee->payroll_type,
                "FechaPago" => $payroll->payed_at->format('Y-m-d'),
                "FechaInicialPago" => $payroll->pay_init->format('Y-m-d'),
                "FechaFinalPago" => $payroll->pay_end->format('Y-m-d'),
                "dias" => $receipt->work_days,
                "totalPercepciones" => round($receipt->salary_total, 2),
                "totalDeducciones" => round($descuento, 2),
                "totalOtrosPagos" => round($receipt->subsidy, 2),
                "numSeguridadSocial" => $receipt->employee->nss,
                "fechaInicioRelLaboral" => $receipt->employee->started_at->format('Y-m-d'),
                "anti" => $ant,
                "tipoRegimen" => $receipt->employee->regime_type,
                "numEmpleado" => $receipt->employee->employee_number,
                "RiesgoPuesto" => $receipt->employee->workstation_risk,                
                "periodicidad" => $receipt->payment_period,
                "salarioDiarioIntegrado" => round($receipt->employee->sdi, 2),
                "claveEntFed" => $receipt->employee->federative_key,
                "totalSueldos" => round($receipt->salary_total, 2),
                "totalGravado" => round($receipt->salary_total, 2),
                "totalExento" => 0,
                "importeGravado" => round($receipt->salary_total, 2),
                "importeExento" => 0,
                "totalOtrasDeducciones" => round($descuento - $receipt->isr , 2),
                "totalImpuestosRetenidos" => round($receipt->isr, 2),
                "importe" => round($descuento, 2),
                "subsidioCausado" => round($receipt->subsidy_c, 2),
            )
        );
        return DocumentServices::getCFDI($data, 'documents/syscfdi.xsl');
    }

    public static function buildInvoiceXMLContent($invoice, $code = null) {
        $concepts = $invoice->invoiceConcepts;
        $respConcepts = [];
        foreach ($concepts as $concept) {
            if ($concept->getDescription()) {
                array_push($respConcepts, [ array (
                    "concepto" => [
                        "ClaveUnidad" => $concept->code,
                        "ClaveProdServ" => $concept->getCode(),
                        "Descripcion" => trim($concept->getDescription(), ' '),
                        "ValorUnitario" => $concept->subtotal,
                        "Importe" => $concept->subtotal,
                        "Impuesto" => $concept->total - $concept->subtotal,
                        "applyRetention" => $invoice->receiver->applyRetention(),
                        "retention" => $concept->retention,
                        "taxRetention" => ($invoice->receiver->retention / 100),
                    ]
                )]);
            }
        }
        if (!$code) {
            $code = 'TEMP';
        }
        $data = array(
            "Folio" => $code,
            "Fecha" => $invoice->invoice_at->format('Y-m-d\TH:i:s'),
            "FormaPago" => $invoice->invoicePayType->code,
            "lugarExpedicion" => $invoice->businessBranch->postal_code,
            "Subtotal" => $invoice->subtotal,
            "Total" => $invoice->calcTotalNeto(),
            "TipoComprobante" => $invoice->invoiceType->code,
            "MetodoPago" => $invoice->pay_method,
            "Emisor" => array (
                "Rfc" => $invoice->emissor->rfc,
                "Nombre" => $invoice->emissor->name,
                "RegimenFiscal" => $invoice->invoiceTaxRegime->code
            ),
            "Receptor" => array(
                "Rfc" => $invoice->receiver->rfc,
                "Nombre" => $invoice->receiver->name,
                "UsoCFDI" => $invoice->invoiceUse->code
            ),
            "Conceptos" => $respConcepts,
            "Impuestos" => array(
                "Importe" => $invoice->total - $invoice->subtotal,
                "applyRetention" => $invoice->receiver->applyRetention(),
                "retention" => $invoice->retention,
            ),
        );
        return DocumentServices::getCFDI($data, 'documents/invoice-cfdi.xsl');
    }

    public static function buildInvoicePaymentXMLContent($invoice, $code = null) {
        $respConcepts = [];
        array_push($respConcepts, [ array(
            "concepto" => [
                'ClaveUnidad' => 'ACT',
                'ClaveProdServ' => '84111506',
                'Cantidad' => 1,
                'Descripcion' => 'Pago'
            ]
        )]);
        $documents = [];
        $paymentAmount = 0;
        foreach ($invoice->payments as $payment) {
            array_push($documents, [ array(
                'documento' => [
                    'IdDocumento' => $payment->invoiceRelated->uuid,
                    'MonedaDR' => 'MXN',
                    'MetodoDePagoDR' => $payment->pay_method,
                    'NumParcialidad' => $payment->parciality,
                    'ImpSaldoAnt' => $payment->last_amount,
                    'ImpPagado' => $payment->amount,
                    'ImpSaldoInsoluto' => $payment->carried_amount
                ]
            ) ]);
            $paymentAmount += $payment->amount;
        }
        
        if (!$code) {
            $code = 'TEMP';
        }
        $data = array(
            "Folio" => $code,
            "Fecha" => $invoice->invoice_at->format('Y-m-d\TH:i:s'),
            "lugarExpedicion" => $invoice->businessBranch->postal_code,
            "Subtotal" => $invoice->subtotal,
            "Total" => $invoice->total,
            "TipoComprobante" => 'P',
            "Emisor" => array (
                "Rfc" => $invoice->emissor->rfc,
                "Nombre" => $invoice->emissor->name,
                "RegimenFiscal" => $invoice->invoiceTaxRegime->code
            ),
            "Receptor" => array(
                "Rfc" => $invoice->receiver->rfc,
                "Nombre" => $invoice->receiver->name,
                "UsoCFDI" => $invoice->invoiceUse->code
            ),
            "Conceptos" => $respConcepts,
            "Complemento" => array(
                "FechaPago" => $invoice->payments[0]->payment_at->format('Y-m-d\TH:i:s'),
                "FormaDePagoP" => $invoice->invoicePayType->code,
                "MonedaP" => 'MXN',
                "Monto" => $paymentAmount
            ),
            "Documentos" => $documents,
        );
        $xml = DocumentServices::getCFDI($data, 'documents/invoice-payment-cfdi.xsl');
        $xml = str_replace('pago100', 'pago10:Pago', $xml);
        $xml = str_replace('pagos100', 'pago10:Pagos', $xml);
        $xml = str_replace('DoctoRelacionado', 'pago10:DoctoRelacionado', $xml);

        return $xml;
    }

    /**
     * Gets a xml content of CFDI
     */
    private static function getCFDI($data, $xsl) {
        $cfdi = DocumentServices::buildCFDI($data, $xsl);
        $domxml = new \DOMDocument('1.0');
        $domxml->preserveWhiteSpace = false;
        $domxml->formatOutput = true;
        $domxml->loadXML($cfdi);
        return $domxml->saveXML();
    }
}
