<?php

namespace App\Http\Services;

use DateTime;
use Idevman\XmlMapper\Support\Facades\XmlMapper;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use SoapClient;
use UnexpectedValueException;
use ZipArchive;

/**
 * This is a value object for BaseConverter containing the sequence
 *
 * NOTE: Changes will not be considering a bracking compatibility change since this utility is for internal usage only
 * @internal
 */
class CertificaCFDI
{

    /**
     * Stamps a payroll receipt
     * @param mixed $xml xml data
     * @param mixed $receipt 
     * @return void 
     * @throws UnexpectedValueException 
     */
    public static function stampPayrollWorkerReceipt($xml, $receipt) {
        CertificaCFDI::stamp($receipt->employee->businessClient, $xml, $receipt->id);
        $error = CertificaCFDI::getError($receipt->id . '_res/Respuesta.xml');
        if ($error) {
            Log::error("Error al timbrar");
            Log::error($error);
            $receipt->log = $error;
            if (strlen($error) > 255) {
                $receipt->log = substr($error, 0, 250);
            }
            $receipt->stamped_at = null;
            $receipt->save();
        } else {
            $receipt->stamped_at = new DateTime();
            $files = Storage::disk('zips')->files($receipt->id . '_res');
            if (count($files) > 0) {
                $map = XmlMapper::mapTo([
                    'TimbreFiscal[uuid]' => 
                        '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[@UUID]',
                ], Storage::disk('zips')->get($files[0]));
                $receipt->uuid = $map['TimbreFiscal']['uuid'];
                $receipt->log = '';
                $receipt->stamp_signer = "SI";
                $receipt->xml_file = $receipt->getStampPath() . $receipt->uuid . ".xml";
                $receipt->save();

                Storage::disk('s3')->put($receipt->xml_file, Storage::disk('zips')->get($files[0]));
            }
        }
        Storage::disk('zips')->deleteDirectory($receipt->id . '_res');
        Storage::disk('zips')->delete($receipt->id . '_res.zip');
        Storage::disk('zips')->delete($receipt->id . '.xml');
        Storage::disk('zips')->delete($receipt->id . '.zip');
    }

    public static function cancelPayrollWorkerReceipt($receipt, $xml) {
        CertificaCFDI::cancel($receipt->employee->businessClient, $xml,
            $receipt->uuid, $receipt->id);

        $files = Storage::disk('zips')->files($receipt->id . '_res');
        if (count($files) > 0) {
            $receipt->canceled_at = new DateTime();
            $receipt->log = '';
            $receipt->cancel_signer = "SI";
            $receipt->save();

            Storage::disk('s3')->put($receipt->getStampPath() . $receipt->uuid . '_cancel.xml', 
                Storage::disk('zips')->get($files[0]));
        }

        Storage::disk('zips')->deleteDirectory($receipt->id . '_res');
        Storage::disk('zips')->delete($receipt->id . '_res.zip');
        Storage::disk('zips')->delete($receipt->id . '.xml');
        Storage::disk('zips')->delete($receipt->id . '.zip');
    }

    /**
     * Stamps a payroll receipt
     * @param mixed $xml xml data
     * @param mixed $payrollReceipt 
     * @return void 
     * @throws UnexpectedValueException 
     */
    public static function stampPayrollReceipt($xml, $payrollReceipt) {
        CertificaCFDI::stamp($payrollReceipt->payroll->businessClient, $xml, 
            $payrollReceipt->id);

        $error = CertificaCFDI::getError($payrollReceipt->id . '_res/Respuesta.xml');
        if ($error) {
            $payrollReceipt->log = $error;
            $payrollReceipt->stamped_at = null;
            $payrollReceipt->save();
        } else {
            $payrollReceipt->stamped_at = new DateTime();
            $files = Storage::disk('zips')->files($payrollReceipt->id . '_res');
            if (count($files) > 0) {
                $map = XmlMapper::mapTo([
                    'TimbreFiscal[uuid]' => 
                        '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[@UUID]',
                ], Storage::disk('zips')->get($files[0]));
                $payrollReceipt->uuid = $map['TimbreFiscal']['uuid'];
                $payrollReceipt->log = '';
                $payrollReceipt->stamp_signer = "SI";
                $payrollReceipt->save();

                Storage::disk('s3')->put($payrollReceipt->getStampPath() . 
                    $payrollReceipt->uuid . ".xml", 
                    Storage::disk('zips')->get($files[0]));
            }
        }

        Storage::disk('zips')->deleteDirectory($payrollReceipt->id . '_res');
        Storage::disk('zips')->delete($payrollReceipt->id . '_res.zip');
        Storage::disk('zips')->delete($payrollReceipt->id . '.xml');
        Storage::disk('zips')->delete($payrollReceipt->id . '.zip');
    }

    /**
     * 
     * Stamps a invoice
     * @param mixed $invoice The data of object invoice
     * @param mixed $xml XML data
     * @return void 
     * @throws UnexpectedValueException 
     */
    public static function stampInvoice($invoice, $xml) {
        CertificaCFDI::stamp($invoice->emissor, $xml, $invoice->id);

        $error = CertificaCFDI::getError($invoice->id . '_res/Respuesta.xml');
        if ($error) {
            $invoice->log = $error;
            $invoice->save();
        } else {
            $invoice->stamped_at = new DateTime();
            $files = Storage::disk('zips')->files($invoice->id . '_res');
            if (count($files) > 0) {
                $map = XmlMapper::mapTo([
                    'TimbreFiscal[uuid]' => 
                        '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[@UUID]',
                ], Storage::disk('zips')->get($files[0]));
                $invoice->uuid = $map['TimbreFiscal']['uuid'];
                $invoice->log = '';
                $invoice->stamp_signer = "SI";
                $invoice->save();

                Storage::disk('s3')->put($invoice->getStampPath() . $invoice->uuid . ".xml", 
                    Storage::disk('zips')->get($files[0]));
            }
        }

        Storage::disk('zips')->deleteDirectory($invoice->id . '_res');
        Storage::disk('zips')->delete($invoice->id . '_res.zip');
        Storage::disk('zips')->delete($invoice->id . '.xml');
        Storage::disk('zips')->delete($invoice->id . '.zip');
    }

    public static function stamp($emissor, $xml, $fileName) {
        $certificated = InvoiceService::certificate($xml, 
            $emissor->code . '/' . $emissor->certificate_file);
        $stamped = InvoiceService::stamp($certificated,
            $emissor->code . '/' . $emissor->key_file,
            $emissor->code . '/' . $emissor->password_file
        );
        Storage::disk('zips')->put("$fileName.xml", $stamped);
        $sRutaCFDI = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '.xml';
        $sNomZip = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '.zip';
            
        $sRutaZipRes = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '_res.zip';
        $sRutaRespuesta = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '_res';

        //Empacar CFDI en ZIP
        //Objeto de la clase ZipArchive
        $oZip = new ZipArchive();
        if ($oZip->open($sNomZip, ZipArchive::CREATE)) {
            $oZip->addFile($sRutaCFDI, $fileName . '.xml');
            $oZip->close();
        }

        //Fin del proceso de empacado
        //Lectura en bytes del CFDI empacado
        $gestor = fopen($sNomZip, "r");
        $contenido= fread($gestor, filesize($sNomZip));
        fclose($gestor);
        // Fin de la lectura
        // Llamada al Web Service
        // Creación del objeto oCliente de la clase SoapClient

        $oCliente = new SoapClient(config('invoicing.CertificaCFDIUrl'));
        //Se crea una variable $aParametros que contiene un array que hace 
        //referencia a los parámetros definidos en el método del Web Service
        $aParametros= array('psUsuario' => config('invoicing.CertificaCFDIUser'),
            'psPassword' => config('invoicing.CertificaCFDIPassword'),
            'pyXML' => $contenido);
        //Se  consume  el  método  getCFDiTest  para  recibir  su  respuesta "getCFDiTestResult"
        if (config('invoicing.mode') == 'test') {
            Log::info('Timbrando en pruebas');
            $aRespuesta = $oCliente->getCFDiTest($aParametros)->getCFDiTestResult;
        } else {
            Log::info('Timbrando en productivo');
            $aRespuesta = $oCliente->getCFDi($aParametros)->getCFDiResult;
        }
        //Fin del llamado al método
        //Se guarda la respuesta obtenida
        $oZipRes= fopen($sRutaZipRes,'w');
        fwrite($oZipRes,$aRespuesta);
        fclose($oZipRes);
        //Se crea un objeto ZIP de la clase ZipArchive
        $oZipR = new ZipArchive();//Se  abre  el  ZIP  para  extraer  los  datos  de  la respuesta  del servicio
        $bRes = $oZipR->open($sRutaZipRes);
        //Se extrae el contenido del Zip
        $oZipR->extractTo($sRutaRespuesta);
        $oZipR->close();

        $error = CertificaCFDI::getError($fileName . "_res/Respuesta.xml");
        if ($error) {
            Log::error('Respuesta de error');
            Log::info(Storage::disk('zips')->get($fileName . "_res/Respuesta.xml"));
            Log::error('XML con certificado');
            Log::info($stamped);

            $body = "Respuesta de error: </br>" . Storage::disk('zips')->get($fileName . "_res/Respuesta.xml") .
                "</br>XML con certificado: </br>" . $stamped;
            $data = array(
                'name'=> '',
                'link' => '',
                'body' => $body,
                'note' => ''
            );
            Mail::send('emails.email', $data, function($message) {
                $message->from('noreply@silifalcon.com', 'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
                $message->subject("Error al timbrar CFDI");
                $message->to('d.maximo@silifalcon.com');
            });
        }
    }

    public static function getError($fileName) {
        if (Storage::disk('zips')->exists($fileName)) {
            $map = XmlMapper::mapTo([
                'Error[codigo,mensaje,detalle]' => 
                    '/soap:Envelop/soap:Body/soap:Fault/Detail/Error[@codigo,@mensaje,@detalle]',
            ], Storage::disk('zips')->get($fileName));
            return 'codigo:' . $map['Error']['codigo'] .
                ' mensaje:' . $map['Error']['mensaje'] .
                ' detalle:' . $map['Error']['detalle'];
        }
        return null;
    }

    /**
     * Cancels a receipt
     * @param mixed $receipt Receipt to cancel
     * @param mixed $xml xml data
     * @return void 
     * @throws UnexpectedValueException 
     */
    public static function cancelPayrollReceipt($receipt, $xml) {
        CertificaCFDI::cancel($receipt->payroll->businessClient, $xml,
            $receipt->uuid, $receipt->id);

        $files = Storage::disk('zips')->files($receipt->id . '_res');
        if (count($files) > 0) {
            $receipt->canceled_at = new DateTime();
            $receipt->log = '';
            $receipt->cancel_signer = "SI";
            $receipt->save();

            Storage::disk('s3')->put($receipt->getStampPath() . $receipt->uuid . '_cancel.xml', 
                Storage::disk('zips')->get($files[0]));
        }

        Storage::disk('zips')->deleteDirectory($receipt->id . '_res');
        Storage::disk('zips')->delete($receipt->id . '_res.zip');
        Storage::disk('zips')->delete($receipt->id . '.xml');
        Storage::disk('zips')->delete($receipt->id . '.zip');
    }

    /**
     * 
     * Stamps a invoice
     * @param mixed $invoice The data of object invoice
     * @param mixed $xml XML data
     * @return void 
     * @throws UnexpectedValueException 
     */
    public static function cancelInvoice($invoice, $xml) {
        CertificaCFDI::cancel($invoice->emissor, $xml, 
            $invoice->uuid, $invoice->id);

        $files = Storage::disk('zips')->files($invoice->id . '_res');
        if (count($files) > 0) {
            $invoice->canceled_at = new DateTime();
            $invoice->cancel_signer = "SI";
            $invoice->log = '';
            $invoice->save();

            Storage::disk('s3')->put($invoice->getStampPath() . $invoice->uuid . '_cancel.xml', 
                Storage::disk('zips')->get($files[0]));
        }

        Storage::disk('zips')->deleteDirectory($invoice->id . '_res');
        Storage::disk('zips')->delete($invoice->id . '_res.zip');
        Storage::disk('zips')->delete($invoice->id . '.xml');
        Storage::disk('zips')->delete($invoice->id . '.zip');
    }

    public static function cancel($emissor, $xml, $uuid, $fileName) {
        $certificated = InvoiceService::certificate($xml, 
            $emissor->code . '/' . $emissor->certificate_file);
        $stamped = InvoiceService::stamp($certificated,
            $emissor->code . '/' . $emissor->key_file,
            $emissor->code . '/' . $emissor->password_file
        );
        Storage::disk('zips')->put($fileName . '.xml', $stamped);

        $sRutaCFDI = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '.xml';
        $sNomZip = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '.zip';
            
        $sRutaZipRes = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '_res.zip';
        $sRutaRespuesta = Storage::disk('zips')->getDriver()
            ->getAdapter()->getPathPrefix() . $fileName . '_res';

        //Empacar CFDI en ZIP
        //Objeto de la clase ZipArchive
        $oZip = new ZipArchive();
        if ($oZip->open($sNomZip, ZipArchive::CREATE)) {
            $oZip->addFile($sRutaCFDI, $fileName . '.xml');
            $oZip->close();
        }

        //Fin del proceso de empacado
        //Lectura en bytes del CFDI empacado
        $gestor = fopen($sNomZip, "r");
        $contenido= fread($gestor, filesize($sNomZip));
        fclose($gestor);
        // Fin de la lectura
        // Llamada al Web Service
        // Creación del objeto oCliente de la clase SoapClient

        error_reporting(E_ALL & ~E_DEPRECATED);
        date_default_timezone_set("America/Mexico_City");
        ini_set("display_errors", 1);

        $pfx = CertificaCFDI::getPfx($emissor);

        $oCliente = new SoapClient(config('invoicing.CertificaCFDIUrl'));
        //Se crea una variable $aParametros que contiene un array que hace 
        //referencia a los parámetros definidos en el método del Web Service
        $aParametros= array('psUsuario' => config('invoicing.CertificaCFDIUser'),
            'psPassword' => config('invoicing.CertificaCFDIPassword'),
            'psRFC' => $emissor->rfc,
            'plstUUID' => array($uuid),
            'pyPFX' => base64_encode($pfx),
            'psPassPFX' => Storage::disk('cert')->get($emissor->code 
                . '/' . $emissor->password_file)
        );
        //Se  consume  el  método  getCFDiTest  para  recibir  su  respuesta "getCFDiTestResult"
        if (config('invoicing.mode') == 'test') {
            Log::info('Cancelando en pruebas');
            $aRespuesta = $oCliente->cancelaCFDiTest($aParametros)->cancelaCFDiTestResult;
        } else {
            Log::info('Cancelando en productivo');
            $aRespuesta = $oCliente->cancelaCFDi($aParametros)->cancelaCFDiResult;
        }
        //Fin del llamado al método
        //Se guarda la respuesta obtenida
        $oZipRes= fopen($sRutaZipRes,'w');
        fwrite($oZipRes,$aRespuesta);
        fclose($oZipRes);
        //Se crea un objeto ZIP de la clase ZipArchive
        $oZipR = new ZipArchive();//Se  abre  el  ZIP  para  extraer  los  datos  de  la respuesta  del servicio
        $bRes = $oZipR->open($sRutaZipRes);
        //Se extrae el contenido del Zip
        $oZipR->extractTo($sRutaRespuesta);
        $oZipR->close();
    }

    public static function getPfx($emissor) {
        if (Storage::disk('cert')
            ->exists($emissor->code . '/pfx.pfx')) {
            return Storage::disk('cert')
                ->get($emissor->code . '/pfx.pfx');
        }
        $storagePath  = Storage::disk('cert')->getDriver()
            ->getAdapter()->getPathPrefix() . $emissor->code . '/';
        $pwd = Storage::disk('cert')->get($emissor->code 
            . '/' . $emissor->password_file);

        $command = 'openssl pkcs8 -inform der -in ' 
            . $storagePath . $emissor->key_file 
            . ' -passin pass:' . $pwd
            . ' -out ' . $storagePath . 'key.pem';
        Log::info($command);
        $output = [];
        exec ($command, $output, $response);

        $command = 'openssl x509 -inform der -in ' 
            . $storagePath . $emissor->certificate_file 
            . ' -out ' . $storagePath . 'cer.pem';
        Log::info($command);
        $output = [];
        exec ($command, $output, $response);

        $command = 'openssl pkcs12 -export ' 
            . '-in ' . $storagePath . 'cer.pem ' 
            . '-inkey ' . $storagePath . 'key.pem ' 
            . '-out ' . $storagePath . 'pfx.pfx ' 
            . '-password pass:' . $pwd;
        Log::info($command);
        $output = [];
        exec ($command, $output, $response);

        return CertificaCFDI::getPfx($emissor);
    }
}