<?php

namespace App\Http\Services;

use App\Model\IsrPeriod;
use App\Model\IsrPeriodDetails;
use App\Model\SubsidyDetail;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

/**
 * This is a value object for BaseConverter containing the sequence
 *
 * NOTE: Changes will not be considering a bracking compatibility change since this utility is for internal usage only
 * @internal
 */
class CFDIService
{
    /**
     * Involve a employee to response format
     */
    public static function getAssimilatedEmployeeResponse($employee, $subsideTable) {
        $subtotal = $employee['salary'];

        $lowerLimits = $subsideTable['lowerLimits'];
        $uperLimits = $subsideTable['upperLimits'];
        $fixedFees = $subsideTable['fixedFees'];
        $surplusLowerLimits = $subsideTable['percents'];
        
        $lowerLimit = 0;
        $percentLowerLimit = 0;
        $fixedFee = 0;
        for ($i = 0; $i < count($lowerLimits); $i++) {
            if ($lowerLimits[$i] <= $subtotal) {
                $lowerLimit = $lowerLimits[$i];
                $percentLowerLimit = $surplusLowerLimits[$i];
                $fixedFee = $fixedFees[$i];
            } else {
                break;
            }
        }
        $surplus = abs($lowerLimit - $subtotal);
        $impMarg = $surplus * $percentLowerLimit;
        $impRet = $impMarg + $fixedFee;
        
        return array(
            'uid' => uniqid(),
            'no' => $employee['no'],
            'name' => $employee['name'],
            'rfc' => $employee['rfc'],
            'curp' => $employee['curp'],
            'workDays' => $employee['workDays'],
            'subtotal' => round($subtotal, 2),
            'isr' => round($impRet, 2),
            'total' => round(($subtotal - $impRet), 2)
        );
    }

    /**
     * Involve a employee to response format
     */
    public static function getSalariedEmployeeResponse($employee, $subsideTable) {
        $sdi = $employee['salaryDay'] * $employee['factory'];
        $import = $employee['salaryDay'] * $employee['days'];
        
        $isr = round(CFDIService::getISR($subsideTable, $import), 2);
        
        $subsidy = 0;
        if ($isr < 0) {
            $subsidy = -$isr * $employee['days'];
        }
        $total = $import + $subsidy;
        $imss = CFDIService::getIMSS($employee, $sdi);
        $isr = CFDIService::getISR($subsideTable, $import);
        $infonavit = CFDIService::getInfonavit($employee);
        $totalPayed = $total - $imss - $isr - $infonavit;
        
        return [
            'uid' => uniqid(),
            'no' => $employee['no'],
            'name' => $employee['name'],
            'rfc' => $employee['rfc'],
            'curp' => $employee['curp'],
            'salaryDay' => $employee['salaryDay'],
            'workDays' => $employee['days'],
            'sdi' => round($sdi, 2),
            'import' => round($import, 2),
            'subsidy' => round($subsidy, 2),
            'total' => round($total, 2),
            'imss' => round($imss, 2),
            'isr' => round($isr, 2),
            'infonavit' => round($infonavit, 2),
            'totalPayed' => round(($totalPayed), 2)
        ];
    }

    /**
     * Calcs the isr on base to total neto
     */
    public static function getIsrForAssimilatedFromTotal($total, $periodId) {
        $isrDetail = IsrPeriodDetails::where('isr_period_id', $periodId)
                ->whereRaw($total . ' + fixed_fee 
                    BETWEEN lower_limit AND 
                    IFNULL(uper_limit,' . $total . ' + fixed_fee)')
                ->get()->first();
        if ($isrDetail->uper_limit) {
            $isr = CFDIService::calcIsr($isrDetail->uper_limit, $isrDetail);
            if ($isrDetail->uper_limit - $isr < $total) {
                $isrDetail = IsrPeriodDetails::where('isr_period_id', $periodId)
                        ->where('lower_limit', '>', $isrDetail->lower_limit)
                        ->orderBy('lower_limit')
                        ->get()->first();
            }
        }

        if (!$isrDetail->uper_limit) {
            $isrDetail->uper_limit = $total * 2; //+ $isrDetail->lower_limit + $isrDetail->fixed_fee;
        }
        $subtotal = $isrDetail->uper_limit;
        $diff = $isrDetail->uper_limit - ($total + $isrDetail->fixed_fee);
        $isr = CFDIService::calcIsr($subtotal, $isrDetail);
        $totalTemp = $subtotal - $isr;
        
        while (round(abs($totalTemp - $total), 2) >= 0.01 && $diff >= 0.01) {
            $diff = round($diff / 2, 2);
            $subtotal -= $diff;
            $isr = CFDIService::calcIsr($subtotal, $isrDetail);
            
            $totalTemp = $subtotal - $isr;
            if ($totalTemp < $total) {
                $subtotal = $subtotal + $diff;
            }
        }
        return $isr;
    }

    /**
     * Calcs the isr on base to total neto
     */
    public static function calcSubsidy($total, $periodId) {
        $isrDetail = IsrPeriodDetails::where('isr_period_id', $periodId)
                ->whereRaw($total . ' BETWEEN lower_limit AND IFNULL(uper_limit,' . $total . ')')
                ->get()->first();
        $subsidyDetail = SubsidyDetail::where('isr_period_id', $periodId)
                ->whereRaw($total . ' BETWEEN lower_limit AND IFNULL(uper_limit,' . $total . ')')
                ->get()->first();
        $diff = $total - $isrDetail->lower_limit;
        $c113 = ($diff * $isrDetail->percent / 100) + $isrDetail->fixed_fee;
        $subsidy = round(($c113 - $subsidyDetail->subsidy) * -1, 2);
        return $subsidy;
    }

    /**
     * Calcs the isr on base to total neto
     */
    public static function calcSubsidyCau($total, $salary_day, $periodId, $workDays) {
        $period = IsrPeriod::find($periodId);
        
        if ($period->days == 1) {
            $subsidyDetail = SubsidyDetail::where('isr_period_id', $periodId)
                ->whereRaw($salary_day . ' BETWEEN lower_limit AND IFNULL(uper_limit,' . $salary_day . ')')
                ->get()->first();
            return $subsidyDetail->subsidy * $workDays;
        }
        $subsidyDetail = SubsidyDetail::where('isr_period_id', $periodId)
                ->whereRaw($total . ' BETWEEN lower_limit AND IFNULL(uper_limit,' . $total . ')')
                ->get()->first();
        return $subsidyDetail->subsidy;
    }


    public static function calcInfonavit($type, $value, $umi, $insurance, $date, $days, $sdi) {
        $current = new Carbon($date);
        $month = $current->month;
        if ($month % 2 == 0) {
            $month --;
        }
        $from = Carbon::createFromDate($current->year, $month, 1, 0, 0, 0);
        $to = Carbon::createFromDate($current->year, $month + 1, 1, 0, 0, 0);
        $to = $to->addDays($to->daysInMonth - 1);
        
        $totalDays = $from->diff($to)->days + 1;
        if ($type == 'F') {
            $importe = $umi * $value * 2;
            $total = $importe + $insurance;
            $factor = round($total / $totalDays, 4);
            return round($factor * $days, 2);
        } else if ($type == 'P') {
            $importe = $sdi * $value / 100;
            $retencion = $importe * $totalDays;
            return round(($retencion + $insurance) / $days, 2);
        } else if ($type == 'C') {
            $importe = $value * 2;
            return round(($importe + $insurance) / $totalDays * $days, 2);
        }
        return 0;
    }

    public static function calcIMSS($sdi, $workDays, $uma) {
        $retImss = $sdi * $workDays * 0.0125;
        $retRCV = $sdi * $workDays * 0.01125;
        $imss = $retImss + $retRCV;
        if ($sdi > $uma * 3) {
            $retRCV = ($sdi - ($uma * 3)) * 0.004 * $workDays;
            $imss += $retRCV;
        }
        return $imss;
    }

    /**
     * Calcs isr from isrDetail
     */
    private static function calcIsr($total, $isr) {
        return round(($total - $isr->lower_limit) * ($isr->percent / 100)
                + $isr->fixed_fee, 2);
    }

    /**
     * Calcs the isr on base to total
     */
    public static function getIsrForAssimilated($total, $periodId) {
        $isr = IsrPeriodDetails::where('isr_period_id', $periodId)
                ->whereRaw($total . ' BETWEEN lower_limit AND ifnull(uper_limit,' . $total . ')')
                ->get()->first();
        
        return CFDIService::calcIsr($total, $isr);
    }

    /**
     * Gets a xml content of CFDI
     */
    public static function getCFDI($data) {
        $subtotal = $data['nomina']['subtotal'];
        $lowerLimits = $data['isr']['LimiteInferior'];
        $uperLimits = $data['isr']['LimiteSuperior'];
        $fixedFees = $data['isr']['CuotaFija'];
        $surplusLowerLimits = $data['isr']['ExcedenteLimiteInferior'];
        
        $lowerLimit = 0;
        $percentLowerLimit = 0;
        $fixedFee = 0;
        for ($i = 0; $i < count($lowerLimits); $i++) {
            if ($lowerLimits[$i] <= $subtotal) {
                $lowerLimit = $lowerLimits[$i];
                $percentLowerLimit = $surplusLowerLimits[$i];
                $fixedFee = $fixedFees[$i];
            } else {
                break;
            }
        }
        $surplus = abs($lowerLimit - $subtotal);
        $impMarg = $surplus * $percentLowerLimit;
        $impRet = $impMarg + $fixedFee;
        $isr = $subtotal - $impRet;

        unset($data['isr']);
        $data['nomina']['isr'] = round($isr, 2);
        $data['nomina']['total'] = round($isr + $subtotal, 2);

        $cfdi = DocumentServices::buildCFDI($data);
        $domxml = new \DOMDocument('1.0');
        $domxml->preserveWhiteSpace = false;
        $domxml->formatOutput = true;
        $domxml->loadXML($cfdi);
        return $domxml->saveXML();
     }

     /**
      * Calculates infonavit deduction
      */
     public static function getInfonavit($employee) {
        if ($employee['workDays']) {
            $import = $employee['SMG'] * $employee['VSM'] * 2;
            $total = $import + $employee['safe'];

            $factor = $total / $employee['bimesterDay'];
            $retention = $factor * $employee['workDays'];
            $infonavit = $retention / 4;
            
            return $infonavit;
        } else {
            $import = $employee['SMG'] * $employee['VSM'];
            $total = $import + $employee['safe'];
            $infonavit = $total / 4;

            return $infonavit;
        }
     }

     /**
     * Calculates imss deduction
     */
    public static function getIMSS($employee, $sdi) {
        $imss = 0;
        if ($sdi > 84.49*3) {
            $imss = ($sdi - (84.49 * 3)) * $employee['days'] * 0.004;
        }
        $imss += ($sdi * 0.0025 * $employee['days'])
                + ($sdi * 0.00375 * $employee['days']) 
                + ($sdi * 0.00625 * $employee['days']) 
                + ($sdi * 0.01125 * $employee['days']);
        return $imss;
    }

    /**
     * Calculates ISR deduction
     */
    public static function getISR($subsideTable, $subtotal) {
        $lowerLimits = $subsideTable['lowerLimits'];
        $uperLimits = $subsideTable['upperLimits'];
        $fixedFees = $subsideTable['fixedFees'];
        $surplusLowerLimits = $subsideTable['percents'];

        $incomes = $subsideTable['incomes'];
        $subsides = $subsideTable['subsides'];

        $lowerLimit = 0;
        $percentLowerLimit = 0;
        $fixedFee = 0;
        for ($i = 0; $i < count($lowerLimits); $i++) {
            if ($lowerLimits[$i] <= $subtotal) {
                $lowerLimit = $lowerLimits[$i];
                $percentLowerLimit = $surplusLowerLimits[$i];
                $fixedFee = $fixedFees[$i];
            } else {
                break;
            }
        }
        $surplus = abs($lowerLimit - $subtotal);
        $impMarg = $surplus * $percentLowerLimit;
        $impRet = $impMarg + $fixedFee;
        $isr = $surplus * $percentLowerLimit + $fixedFee;

        $subside = 0;
        for ($i = 0; $i < count($incomes); $i++) {
            if ($subtotal > $incomes[$i]) {
                $subside = $subsides[$i];
            } else {
                break;
            }
        }
        return $isr - $subside;
    }
}