<?php

namespace App\Http\Controllers;

use App\Http\Services\FacturacionModerna;
use App\Http\Services\InvoiceService;
use DateTime;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class StampController extends Controller {
    
    public function cancelInvoice($invoice) {
        Log::info('Cancelando: ' . $invoice->uuid);
        $url = config('invoicing.stampingUrl');
        $user_id = config('invoicing.webServiceUser');
        $user_password = config('invoicing.webServicePassword');

        $parametros = array(
            'emisorRFC' => $invoice->emissor->rfc,
            'UserID' => $user_id,
            'UserPass' => $user_password
        );

        $debug = 1;
        $cliente = new FacturacionModerna($url, $parametros, $debug);
        try {
            if ($cliente->cancelar($invoice->uuid, null)) {
                Log::info("Cancelación exitosa");
                $invoice->canceled_at = new DateTime();
                $invoice->log = '';
                $invoice->cancel_signer = 'FM';
                $invoice->save();
            } else {
                $log = "[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError;
                Log::error($log);
                if (strlen($log) > 250) {
                    $log = substr($log, 0, 250);
                }
                $invoice->log = $log;
                $invoice->canceled_at = null;
                $invoice->save();
            }
        } catch (Exception $ex) {
            log::error($ex);
            $log = $ex->getMessage();
            if (strlen($log) > 250) {
                $log = substr($log, 0, 250);
            }
            $invoice->log = $log;
            $invoice->canceled_at = null;
            $invoice->save();
        }
    }

    public function stampInvoice($xml, $invoice) {
        $businessClient = $invoice->emissor;
        $certificated = InvoiceService::certificate($xml, 
            $businessClient->code . '/' . $businessClient->certificate_file);
        
        $stamped = InvoiceService::stamp($certificated, 
            $businessClient->code . '/' . $businessClient->key_file,
            $businessClient->code . '/' . $businessClient->password_file
        );

        $opciones = array();  
        $opciones['generarCBB'] = true;
        $opciones['generarPDF'] = true;
        $opciones['generarTXT'] = false;

        $url_timbrado = config('invoicing.stampingUrl');
        $user_id = config('invoicing.webServiceUser');
        $user_password = config('invoicing.webServicePassword');

        $parametros = array(
            'emisorRFC' => $businessClient->rfc,
            'UserID' => $user_id,
            'UserPass' => $user_password
        );

        $debug = 1;
        $cliente = new FacturacionModerna($url_timbrado, $parametros, $debug);
        try {
            $invoice->stamped_at = new DateTime();
            if ($cliente->timbrar($stamped, $opciones)) {
                if ($cliente->xml) {
                    Storage::disk('s3')->put($invoice->getStampPath() . $cliente->UUID . ".xml", $cliente->xml);
                }
                if (isset($cliente->pdf)) {
                    Storage::disk('s3')->put($invoice->getStampPath() . $cliente->UUID . ".pdf", $cliente->pdf);
                }
                if (isset($cliente->png)) {
                    Storage::disk('s3')->put($invoice->getStampPath() . $cliente->UUID .".png", $cliente->png);
                }
                $invoice->uuid = $cliente->UUID;
                $invoice->log = '';
                $invoice->stamp_signer = 'FM';
                $invoice->save();

                Log::info("Timbrado exitoso");
            } else {
                $log = "[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError;
                if (strlen($log) > 250) {
                    $log = substr($log, 0, 250);
                }
                $invoice->log = $log;
                $invoice->stamped_at = null;
                $invoice->save();
                Log::error("[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError);
            }
        } catch (Exception $ex) {
            Log::error($ex);
            $log = $ex->getMessage();
            if (strlen($log) > 250) {
                $log = substr($log, 0, 250);
            }
            $invoice->log = $log;
            $invoice->stamped_at = null;
            $invoice->save();
        }
    }

    public function cancelPayrollReceipt($payrollReceipt) {
        Log::info('Cancelando: ' . $payrollReceipt->uuid);
        $url = config('invoicing.stampingUrl');
        $user_id = config('invoicing.webServiceUser');
        $user_password = config('invoicing.webServicePassword');

        $parametros = array(
            'emisorRFC' => $payrollReceipt->payroll->rfc,
            'UserID' => $user_id,
            'UserPass' => $user_password
        );

        $debug = 1;
        $cliente = new FacturacionModerna($url, $parametros, $debug);
        try {
            if ($cliente->cancelar($payrollReceipt->uuid, null)) {
                Log::info("Cancelación exitosa");
                $payrollReceipt->canceled_at = new DateTime();
                $payrollReceipt->log = '';
                $payrollReceipt->cancel_signer = 'FM';
                $payrollReceipt->save();
            } else {
                $log = "[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError;
                Log::error($log);
                if (strlen($log) > 250) {
                    $log = substr($log, 0, 250);
                }
                $payrollReceipt->log = $log;
                $payrollReceipt->canceled_at = null;
                $payrollReceipt->save();
            }
        } catch (Exception $ex) {
            log::error($ex);
            $log = $ex->getMessage();
            if (strlen($log) > 250) {
                $log = substr($log, 0, 250);
            }
            $payrollReceipt->log = $log;
            $payrollReceipt->canceled_at = null;
            $payrollReceipt->save();
        }
    }

    public function stampPayrollReceipt($xml, $payrollReceipt) {
        $businessClient = $payrollReceipt->payroll->businessClient;
        $certificated = InvoiceService::certificate($xml, 
            $businessClient->code . '/' . $businessClient->certificate_file);
        
        $stamped = InvoiceService::stamp($certificated, 
            $businessClient->code . '/' . $businessClient->key_file,
            $businessClient->code . '/' . $businessClient->password_file
        );

        $opciones = array();  
        $opciones['generarCBB'] = true;
        $opciones['generarPDF'] = true;
        $opciones['generarTXT'] = false;

        $url_timbrado = config('invoicing.stampingUrl');
        $user_id = config('invoicing.webServiceUser');
        $user_password = config('invoicing.webServicePassword');

        $parametros = array(
            'emisorRFC' => $payrollReceipt->payroll->rfc,
            'UserID' => $user_id,
            'UserPass' => $user_password
        );

        $debug = 1;
        $cliente = new FacturacionModerna($url_timbrado, $parametros, $debug);
        try
        {
            $payrollReceipt->stamped_at = new DateTime();
            if ($cliente->timbrar($stamped, $opciones)) {
                if ($cliente->xml) {
                    Storage::disk('s3')->put($payrollReceipt->getStampPath() . $cliente->UUID . ".xml", $cliente->xml);
                }
                if (isset($cliente->pdf)) {
                    Storage::disk('s3')->put($payrollReceipt->getStampPath() . $cliente->UUID . ".pdf", $cliente->pdf);
                }
                if (isset($cliente->png)) {
                    Storage::disk('s3')->put($payrollReceipt->getStampPath() . $cliente->UUID .".png", $cliente->png);
                }
                $payrollReceipt->uuid = $cliente->UUID;
                $payrollReceipt->log = '';
                $payrollReceipt->stamp_signer = 'FM';
                $payrollReceipt->save();

                Log::info("Timbrado exitoso");
            } else {
                $log = "[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError;
                if (strlen($log) > 250) {
                    $log = substr($log, 0, 250);
                }
                $payrollReceipt->log = $log;
                $payrollReceipt->stamped_at = null;
                $payrollReceipt->save();
                Log::error("[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError);
            }
        } catch (Exception $ex) {
            Log::error($ex);
            $log = $ex->getMessage();
            if (strlen($log) > 250) {
                $log = substr($log, 0, 250);
            }
            $payrollReceipt->log = $log;
            $payrollReceipt->stamped_at = null;
            $payrollReceipt->save();
        }
    }

    public function stamp(Request $request, $id, $rfc) {
        if (Storage::disk('s3')->exists($id . '.xml')) {
            return response(Storage::disk('s3')->get($id . '.xml'), 200, [
                'Content-Type' => 'text/xml',
                'Content-Disposition' => 'attachment; filename="' .$id . '.xml"',
            ]);
        } else {
            $xml = Storage::disk('local')->get($id . '.xml');
            $certificated = InvoiceService::certificate($xml, 'FAC_MODE/certificate.cer');
            
            $stamped = InvoiceService::stamp($certificated,
                'FAC_MOD/private.key',
                'FAC_MOD/password.txt'
            );

            $opciones = array();  
            $opciones['generarCBB'] = true;
            $opciones['generarPDF'] = true;
            $opciones['generarTXT'] = false;

            $url_timbrado = config('invoicing.stampingUrl');
            $user_id = config('invoicing.webServiceUser');
            $user_password = config('invoicing.webServicePassword');

            $parametros = array(
                'emisorRFC' => $rfc,
                'UserID' => $user_id,
                'UserPass' => $user_password
            );

            $debug = 1;
            $cliente = new FacturacionModerna($url_timbrado, $parametros, $debug);
            if ($cliente->timbrar($stamped, $opciones)) {
                $comprobante = 'comprobantes/'.$cliente->UUID;
                if ($cliente->xml) {
                    Storage::disk('s3')->put($id . ".xml", $cliente->xml);
                }
                if (isset($cliente->pdf)) {
                    Storage::disk('s3')->put($id . ".pdf", $cliente->pdf);
                }
                if (isset($cliente->png)) {
                    Storage::disk('s3')->put($id .".png", $cliente->png);
                }
                Log::info("Timbrado exitoso");
            } else {
                Log::error("[".$cliente->ultimoCodigoError."] - ".$cliente->ultimoError);
            }

            return response(Storage::disk('local')->get($id . '.xml'), 200, [
                'Content-Type' => 'text/xml',
                'Content-Disposition' => 'attachment; filename="' .$id . '.xml"',
            ]);
        }
    }

}
