<?php

namespace App\Http\Controllers\Requests;

use App\Http\Controllers\Controller;
use App\Http\Services\CodeService;
use App\Model\BusinessClient;
use App\Model\EmployeeScatter;
use App\Model\Request as ModelRequest;
use App\Model\RequestDeposit;
use App\Model\RequestInvoice;
use App\Model\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\MessageBag;
use Illuminate\Support\Facades\Mail;

class RequestController extends Controller
{
    /**
     * List all expedients in page
     */
    public function index(Request $request) {
        $partnership = Auth::user()->partnership();
        
        $page = $request->input('page', 0);
        $search = $request->input('search');
        $sort = 'requests.id';
        $sortOrientation = 'asc';
        $sortTokens = $request->input('sort', 'requests.id:desc');
        if (strpos($sortTokens, ':') != false) {
            list($sort, $sortOrientation) = explode(':', $sortTokens);
        } else {
            $sort = $sortTokens;
        }

        $query = ModelRequest::join('business_clients', 'business_clients.id', '=', 'requests.business_client_id')
            ->where('business_clients.partnership_id', $partnership->id)
            ->select('requests.*')
            ->orderBy($sort, $sortOrientation);

        if ($search) {
            $query = $query->where(function ($query) use($search) {
                $query->where('business_clients.name', 'like', '%' . $search . '%');
            });
        }
        $total = $query->count();
        return view('client.request-index', [
            'data' => $query->skip($page * 25)->take(25)->get(),
            'total' => $total,
            'page' => $page,
            'search' => $search,
            'sort' => $sort,
            'sortOrientation' => $sortOrientation,
            'pages' => ceil($total / 25)
        ]);
    }

    /**
     * Shows a form for create a new client request
     * @param Request $request 
     * @return void 
     */
    public function create(Request $request) {
        $partnership = Auth::user()->partnership();
        $record = new ModelRequest();
        $internalClients = BusinessClient::where('type', 'I')
            ->where('status', 'R')
            ->whereNull('canceled_at')->get();
        
        $deposits = [];
        $invoice = new RequestInvoice();
        $invoices = [$invoice];
        $concepts = [];
        return view('client.new-request', [
            'record' => $record,
            'partnership' => $partnership,
            'internalClients' => $internalClients,
            'newDeposit' => 1,
            'newConcept' => 1,
            'editInvoice' => $request->has('edit-invoice') ? 
                $request->input('edit-invoice') : 0,
            'deposits' => $deposits,
            'invoices' => $invoices,
            'concepts' => $concepts
        ]);
    }

    /**
     * Edits a request model
     */
    public function edit(Request $request, $id) {
        $record = ModelRequest::find($id);
        $partnership = $record->businessClient->partnership;
        
        $internalClients = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')->get();
        
        $deposits = $record->deposits;
        $invoices = [];
        foreach ($record->invoices as $invoice) {
            array_push($invoices, $invoice);
        }
        return view('client.new-request', [
            'record' => $record,
            'partnership' => $partnership,
            'internalClients' => $internalClients,
            'newDeposit' => $request->has('new-deposit') ? $request->input('new-deposit') : 0,
            'deposits' => $deposits
        ])->withErrors($this->getErrors($request, $record));
    }

    private function getErrors(Request $request, $record) {
        $errors = new MessageBag();
        if ($request->has('edit-invoice')) {
            foreach ($record->invoices as $invoice) {
                if ($invoice->id == $request->input('edit-invoice')
                    && $invoice->amount == 0) {
                    $errors->add('invoice-amount', 'Completa este campo');
                }
            }
        }
        return $errors;
    }

    /**
     * Stores a invoice of a client
     * @param Request $request Request data
     * @return void 
     */
    public function store(Request $request) {
        $record = RequestController::storeRequest($request);
        return redirect('/client/requests/' . $record->id . '/edit');
    }

    /**
     * Stores a invoice of a client
     * @param Request $request Request data
     * @return void 
     */
    public function sendRequest(Request $request) {
        $record = RequestController::storeRequest($request);
        /*
        if (count($record->deposits) == 0) {
            Cache::put('warning', 'Por favor agregue por lo menos un deposito');
            return redirect()->back()->withInput($request->input());
        }

        if (count($record->requestInvoices) == 0) {
            Cache::put('warning', 'Por favor ingrese al menos una factura');
            return redirect()->back()->withInput($request->input());
        }*/

        for ($i = 0; $i < count($record->requestInvoices); $i++) {
            if (count($record->requestInvoices[$i]->concepts) == 0) {
                Cache::put('warning', 'Por favor llene correctamente los conceptos de la factura ' . ($i + 1));
                return redirect('/client/requests/' . $record->id . '/invoices/' . $record->invoices[$i]->id . '/edit');
            }
        }

        /*
        if (count($record->employees) == 0) {
            Cache::put('warning', 'Por favor genere la dispersión de empleados');
            return redirect('/client/requests/' . $record->id . '/scatter');
        }*/

        $scatter = 0;
        $employeeScatters = EmployeeScatter::where('request_id', $request->id)->with('scatterDetails')->get();
        foreach ($employeeScatters as $employeeScatter) {
            $scatter += $employeeScatter->amount + $employeeScatter->getPendingAmount();
        }
        //$scatter = DB::table("employee_scatters")->where('request_id', $record->id)->get()->sum("amount");
        if (round($scatter, 2) > round($record->operation, 2)) {
            Cache::put('warning', 'El total de la dispersión debe ser $ ' . number_format($record->operation, 2));
            return redirect('/client/requests/' . $record->id . '/scatter');
        }

        if (count($record->requestInvoices) > 0) {
            $amount = 0;
            for ($i = 0; $i < count($record->requestInvoices); $i++) {
                $amount += $record->requestInvoices[$i]->amount;
            }
            if (round($amount, 2) != $record->total) {
                Cache::put('warning', 'El total de las facturas debe ser $ ' . number_format($record->total, 2));
                return redirect('/client/requests/' . $record->id . '/edit');
            }
        }

        $partnershipCode = Auth::user()->partnership()->code;
        if (!$record->code || $record->code == '') {
            $record->code = CodeService::saveNext("$partnershipCode-", "REQUEST$partnershipCode-");
        }
        if (!$record->isReleased()) {
            $record->status = 'P';
            $record->fac_status = null;
            $record->bank_status = null;
            $record->save();
        }

        RequestController::sendRequestEmail($record);

        $employeeScatters = EmployeeScatter::where('request_id', $request->id)->with('scatterDetails')->get();
        foreach ($employeeScatters as $employeeScatter) {
            $employeeScatter->sendPendingAmount();
        }
        
        Cache::put('message', 'La solicitud se envio correctamente');
        return redirect('/client/requests/' . $record->id . '/edit');
    }

    public static function storeRequest(Request $request) {
        $record = new ModelRequest();
        if ($request->has('id')) {
            $record = ModelRequest::find($request->input('id'));
        }
        $partnership = Auth::user()->partnership();
        if ($request->input('id') != null) {
            $record = ModelRequest::find($request['id']);
        }

        if (!$record->isReleased()) {
            $record->status = 'O';
            $record->user_id = Auth::id();
            $record->total = $request->input('total');
            $record->operation = $request->input('operation');
            $record->business_client_id = $request->input('client');
            if (!$record->scatter_file) {
                $record->scatter_file = '';
            }
            $record->comments = $request->input('comments');
            $record->released_at = Carbon::now();
            $record->save();
        }

        if ($request->has('origin-client') && $request->has('destination-client')) {
            $deposit = new RequestDeposit();
            $deposit->request()->associate($record);
            $deposit->origin_id = $request->input('origin-client');
            $deposit->destination_id = $request->input('destination-client');
            if ($request->has('voucher-file')) {
                $file = $request->file('voucher-file');
                $deposit->voucher_file = $partnership->code . '/requests/' . $record->id . '/' . $file->getClientOriginalName();

                $existFile = false;
                $deposits = $record->deposits;
                foreach ($deposits as $dep) {
                    if ($dep->voucher_file == $deposit->voucher_file) {
                        $existFile = true;
                        break;
                    }
                }
                if (!$existFile) {
                    $voucher = file_get_contents($file->getRealPath());
                    Storage::disk('s3')->put($deposit->voucher_file, $voucher);
                } else {
                    Cache::put('warning', 'Ya existe un deposito con el mismo nombre por favor seleccione otro');
                    return $record;
                }
            } else {
                $deposit->voucher_file = '';
            }
            $deposit->save();
        }

        $addFile = false;
        if ($request->has('edit-voucher-ids')) {
            $editFilesId = $request->input('edit-voucher-ids');            
            for ($i = 0; $i < count($editFilesId); $i++) {
                $id = $editFilesId[$i];
                if ($request->has("edit-voucher-files-$id")) {
                    $deposit = RequestDeposit::find($id);
                    if ($deposit) {
                        $file = $request->file("edit-voucher-files-$id");
                        $deposit->voucher_file = $partnership->code . '/requests/' . $record->id . '/' . $file->getClientOriginalName();
                        $voucher = file_get_contents($file->getRealPath());
                        Storage::disk('s3')->put($deposit->voucher_file, $voucher);
                        $deposit->save();
                        $addFile = true;
                    }
                }
            }
        }

        if ($addFile) {
            RequestController::sendRequestEmail($record);
        }
        return $record;
    }

    public static function sendRequestEmail($record) {
        $data = array(
            'user'=> Auth::user(),
            'partnership' => Auth::user()->partnership(),
            'record' => $record
        );
        
        $users = User::select('users.email')
            ->join('user_authorizations', 'users.id', '=', 'user_authorizations.user_id')
            ->join('authorizations', 'authorizations.id', '=', 'user_authorizations.authorization_id')
            ->where('authorizations.code', 'SCATTER_REQUESTS')
            ->get();
            /*
        $users = User::select('users.email')
            ->where('id', 1)
            ->get();
            */
        
        Mail::send('emails.client-request', $data, function($message) use ($users, $record) {
            $message->from('noreply@silifalcon.com', 'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
            $message->subject("Nueva solicitud de dispersión de empleados:" . $record->code);
            foreach ($users as $user) {
                $email = str_replace(" ", "", $user->email);
                if ($email != '' && strpos($email, '@') !== false) {
                    $message->to(str_replace(" ", "", $email));
                }
            }
        });
    }
}
