<?php

namespace App\Http\Controllers;

use App\Model\ProcessRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class ProcessRequestController extends Controller {

    /**
     * Execute command
     * @param request Controller request
     */
    public function command(Request $request) {
        if ($request->input('action') === 'send-email') {
            return $this->sendEmail($request);
        } else if ($request->input('action') === 'deleteExport') {
            return $this->removePayrolls($request);
        } else if ($request->input('action') === 'search-page') {
            return $this->loadPayrollDownloads($request);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Loads download controller
     */
    public function loadPayrollDownloads(Request $request) {
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }
        $query = ProcessRequest::where('command', 'like', 'bb:export-receipts%')
            ->where('expired_at', '>=', Carbon::now()->startOfDay())
            ->orderBy('id', 'desc');
        $total = $query->count();
        $query->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('process.downloads', [
            'page' => $page,
            'pages' => $pages,
            'process' => $query->get()
        ]);
    }

    /**
     * Loads download controller
     */
    public function loadSysPayrollDownloads(Request $request) {
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }
        $query = ProcessRequest::where('command', 'like', 'bb:export-worker-receipts%')
            ->where('expired_at', '>=', Carbon::now()->startOfDay())
            ->orderBy('id', 'desc');
        $total = $query->count();
        $query->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('process.sys-downloads', [
            'page' => $page,
            'pages' => $pages,
            'process' => $query->get()
        ]);
    }

    public function loadStampedDownloads(Request $request) {
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }
        $query = ProcessRequest::where('command', 'like', 'bb:stamp-assimilated%')
            ->where('expired_at', '>=', Carbon::now()->startOfDay())
            ->orderBy('id', 'desc');
        $total = $query->count();
        $query->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('process.stampeds', [
            'page' => $page,
            'pages' => $pages,
            'process' => $query->get()
        ]);
    }

    public function loadSysStampedDownloads(Request $request) {
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }
        $query = ProcessRequest::where('command', 'like', 'bb:stamp-worker%')
            ->where('expired_at', '>=', Carbon::now()->startOfDay())
            ->orderBy('id', 'desc');
        $total = $query->count();
        $query->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('process.sys-stampeds', [
            'page' => $page,
            'pages' => $pages,
            'process' => $query->get()
        ]);
    }

    /**
     * Loads download controller
     */
    public function sendEmail(Request $request) {
        $emails = explode(',', $request->input('email'));
        $count = 0;
        foreach ($emails as $email) {
            $email = str_replace(" ", "", $email);
            if ($email != '' && strpos($email, '@') !== false) {
                $count ++;    
            }
        }
        if ($count == 0) {
            Cache::put('warning', 'Por favor ingrese al menos un correo destinatario');
            return redirect()->back()->withInput($request->input());    
        }

        $process = ProcessRequest::find($request->input('id-email'));
        $currentUrl = strstr($request->url(), 'download', true) . 'download/' . $process->link_code;
        Log::info('Enviando correo del comando:' . $process->command);
        Log::info($currentUrl);

        $data = array(
            'name'=> 'Buen día, adjunto liga de descarga de los recibos de nómina',
            'link' => $currentUrl,
            'body' => $request->input('body'),
            'note' => 'La liga expira el ' . $process->expired_at->format('d/m/Y') . ' Favor de descargar maximo en esa fecha',
        );

        Log::info('Enviando correo ha:');
        Mail::send('emails.email', $data, function($message) use ($emails) {
            $message->from('noreply@silifalcon.com', 
                'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
            $message->subject('Recibos de nómina de empleados');
            foreach ($emails as $email) {
                $email = str_replace(" ", "", $email);
                if ($email != '' && strpos($email, '@') !== false) {
                    Log::info(str_replace(" ", "", $email));
                    $message->to(str_replace(" ", "", $email));
                }
            }          
        });
        Cache::put('message', 'Se envio el correo');
        return redirect()->back()->withInput($request->input());
    }

    public function delete(Request $request, $id) {
        $process = ProcessRequest::find($id);
        if (Storage::disk('zips')->exists($process->code . '.zip')) {
            Log::info('Eliminando archivo' . $process->code . '.zip');
            Storage::disk('zips')->delete($process->code . '.zip');
        } else {
            Log::info('No se encontro el archivo' . $process->code . '.zip');
        }
        $process->delete();
        Cache::put('message', 'Se removio la descarga: ' . 
            $process->getOptionValue('--payroll'));
        return redirect('/downloads');
    }

    public function list(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);

        $query = ProcessRequest::where('status', 'P')
            ->skip($page * $pageSize)->take($pageSize);
            
        $query->orderBy('created_at', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }
}
