<?php

namespace App\Http\Controllers\Partnership;

use Illuminate\Http\Request;
use App\Model\BusinessClient;
use App\Http\Controllers\Controller;
use App\Model\Partnership;
use App\Model\PartnershipContact;
use Illuminate\Support\Facades\Cache;

class PartnershipController extends Controller
{
    
    /**
     * Retrieves a list data of internal business client
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);

        $query = Partnership::select('*');
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('code', 'like', '%' . $filter . '%')
                    ->orWhere('name', 'like', '%' . $filter . '%');
            });
        }
        
        $total = $query->count('id');
        $query->orderBy('id')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        return view('admin.partnership-index', [
            'partnerships' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages
        ]);
    }

    /**
     * Stores a partnership, filled on form request
     * 
     * @param request The request data with values
     */
    public function store(Request $request) {
        $partnership = new Partnership();
        if ($request['id'] != null) {
            $partnership = Partnership::find($request['id']);
        }
        $partnership->code = strtoupper($request['code']);
        $partnership->name = $request['name'];
        $partnership->prefix = strtoupper($request['prefix']);
        $partnership->save();

        Cache::put('message', 'Se salvo correctamente al consorcio ' . 
            $partnership->name);
        return redirect('/admin/partnerships/' . $partnership->id . '/edit');
    }

    /**
     * Shows a partnership form for add a partnership
     */
    public function create(Request $request) {
        $partnership = new Partnership();
        $contacts = [];
        $businessClients = [];
        return view('admin.partnership', [
            'partnership' => $partnership,
            'contacts' => $contacts,
            'businessClients' => $businessClients,
            'conceptTab' => 'div-business-clients'
        ]);
    }

    /**
     * Shows a partnership form for edit his data
     * 
     * @param request request data
     * @param id The id of partnership
     */
    public function edit(Request $request, $id) {
        $partnership = Partnership::find($id);
        $contacts = PartnershipContact::
            where('partnership_id', $partnership->id)->get();
        $businessClients = BusinessClient::where('partnership_id', $id)->get();
        $internalBusineses = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')
            ->get();
        $businessConceptActive = $internalBusineses[0]->code;
        return view('admin.partnership', [
            'partnership' => $partnership,
            'contacts' => $contacts,
            'businessClients' => $businessClients,
            'internalBusineses' => $internalBusineses,
            'businessConceptActive' => $businessConceptActive,
            'conceptTab' => 'div-business-clients'
        ]);
    }

    /**
     * Shows a partnership form for edit his data
     * 
     * @param request request data
     * @param id The id of partnership
     */
    public function editTab(Request $request, $id, $conceptTab) {
        $partnership = Partnership::find($id);
        $contacts = PartnershipContact::
            where('partnership_id', $partnership->id)->get();
        $businessClients = BusinessClient::where('partnership_id', $id)->get();
        $internalBusineses = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')
            ->get();
        $businessConceptActive = $conceptTab;
        $conceptTab = 'div-concepts';
        return view('admin.partnership', [
            'partnership' => $partnership,
            'contacts' => $contacts,
            'businessClients' => $businessClients,
            'internalBusineses' => $internalBusineses,
            'businessConceptActive' => $businessConceptActive,
            'conceptTab' => $conceptTab
        ]);
    }

    /**
     * Remove a business client from specified business
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un consorcio');
        } else {
            Partnership::whereIn('id', $request->input('selected'))
                ->delete();
            Cache::put('message', 'Los consorcios han sido dados de baja');
        }
        return redirect('/admin/partnerships');
    }

}
