<?php

namespace App\Http\Controllers\Partnership;

use App\Http\Controllers\Controller;
use App\Model\BusinessClient;
use App\Model\Partnership;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class BusinessController extends Controller
{
        
    /**
     * Retrieves a list data of external business client
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = BusinessClient::where('type', 'E')
            ->where('status', 'R')
            ->whereNull('canceled_at');
        if ($status != 'V') {
            $query = BusinessClient::whereNotNull('canceled_at');
        }
        
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->Where('name', 'like', '%' . $filter . '%')
                    ->orWhere('rfc', 'like', '%' . $filter . '%')
                    ->orWhere('address', 'like', '%' . $filter . '%');
            });
        }
        
        $total = $query->count('id');
        $query->orderBy('name')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        return view('admin.external-business-client-index', [
            'businessClients' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'status' => $status
        ]);
    }

    /**
     * Shows a business client form for add a business client
     */
    public function create(Request $request) {
        $business = new BusinessClient();
        $departments = [];
        return view('admin.external-business-client', [
            'business' => $business,
            'departments' => $departments,
            'partnerships' => Partnership::all()
        ]);
    }

    /**
     * Stores a business cliente, filled on form request
     * 
     * @param request The request data with values
     */
    public function store(Request $request) {
        $this->validBusinessClientForm($request);

        $business = new BusinessClient();
        if ($request['id'] != null) {
            $business = BusinessClient::find($request['id']);
        }
        $business->status = 'R';
        $business->code = strtoupper($request['rfc']);
        $business->name = $request['name'];
        $business->rfc = strtoupper($request['rfc']);
        $business->type = 'E';
        $business->address = $request['address'];
        $business->retention = $request['retention'];
        $business->partnership_id = 
            ($request['partnership'] && $request['partnership'] != '') ? 
            $request['partnership'] : null;
        $business->save();

        Cache::put('message', 'Se salvo correctamente al cliente externo ' . 
            $business->name);
        return redirect('/admin/external-business-clients/' . $business->id . '/edit');
    }

    /**
     * Shows a business client form for edit his data
     * 
     * @param request request data
     * @param id The id of business client
     */
    public function edit(Request $request, $id) {
        $business = BusinessClient::find($id);
        return view('admin.external-business-client', [
            'business' => $business,
            'partnerships' => Partnership::all()
        ]);
    }

    /**
     * Remove a business client from specified business
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un cliente externo');
        } else {
            BusinessClient::whereIn('id', $request->input('selected'))
                ->where('type', 'E')
                ->update(['canceled_at' => Carbon::now()]);
            Cache::put('message', 'Los clientes externos han sido dados de baja');
        }
        return redirect('/admin/external-business-clients');
    }

    /**
     * Valid request form
     */
    public function validBusinessClientForm(Request $request) {
        $request->validate([
            'name' => 'required',
            'rfc' => 'required'
        ], [
            'name.required' => 'El nombre es requerido',
            'rfc.required' => 'El rfc es requerido'
        ]);
    }
}
