<?php

namespace App\Http\Controllers;

use App\Http\Services\CodeService;
use App\Model\Bank;
use App\Model\BusinessBranch;
use App\Model\BusinessClient;
use App\Model\BusinessDepartment;
use App\Model\ConfigCodes;
use App\Model\Employee;
use App\Model\Invoice;
use App\Model\IsrPeriod;
use App\Model\IsrPeriodDetails;
use App\Model\Partnership;
use App\Model\Payroll;
use App\Model\PayrollReceipt;
use App\Model\PayrollWorkerReceipt;
use App\Model\ProcessRequest;
use Carbon\Carbon;
use DateTime;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ModelController extends Controller
{
    
    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function saveBusinessClient(Request $request) {
        $record = new BusinessClient();
        if ($request->has('id')) {
            $record = BusinessClient::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('code')) {
            $record->code = $request->json('code');
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        if ($request->has('rfc')) {
            $record->rfc = $request->json('rfc');
        }
        if ($request->has('prefix')) {
            $record->prefix = $request->json('prefix');
        }
        if ($request->has('certificateFile')) {
            $record->certificate_file = $request->json('certificateFile');
        }
        if ($request->has('keyFile')) {
            $record->key_file = $request->json('keyFile');
        }
        if ($request->has('passwordFile')) {
            $record->password_file = $request->json('passwordFile');
        }
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listBusinessClient(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);

        $query = BusinessClient::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        $query->orderBy('created_at', 'desc');
        $query->whereNull('canceled_at');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function cancelBusinessClient(Request $request) {
        $record = null;
        
        $record = BusinessClient::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->canceled_at = new DateTime();
        $record->save();
        
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    public function saveBranch(Request $request) {
        $record = new BusinessBranch();
        if ($request->has('id')) {
            $record = BusinessBranch::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        if ($request->has('prefix')) {
            $record->prefix = $request->json('prefix');
        }
        if ($request->has('postal_code')) {
            $record->postal_code = $request->json('postal_code');
        }
        if ($request->has('businessClient')) {
            $record->business_client_id = $request->json('businessClient');
        }
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listBranchs(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $businessId = $request->input('business', null);

        $query = BusinessBranch::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        if (!empty($businessId)) {
            $query->where('business_client_id', $businessId);
        }
        $query->orderBy('business_client_id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    public function saveBank(Request $request) {
        $record = new Bank();
        if ($request->has('id')) {
            $record = Bank::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listBanks(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);

        $query = Bank::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        $query->orderBy('name', 'asc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function deleteBank(Request $request) {        
        $record = Bank::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->delete();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function deleteBranch(Request $request) {
        $record = null;
        
        $record = BusinessBranch::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->delete();
        
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    public function saveDepartment(Request $request) {
        $record = new BusinessDepartment();
        if ($request->has('id')) {
            $record = BusinessDepartment::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('code')) {
            $record->code = $request->json('code');
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        if ($request->has('email')) {
            $record->email = $request->json('email');
        }
        if ($request->has('businessClient')) {
            $record->business_client_id = $request->json('businessClient');
        }
        if ($request->has('prefix')) {
            $record->prefix = $request->json('prefix');
        }
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listDepartments(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $businessId = $request->input('business', null);

        $query = BusinessDepartment::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        if (!empty($businessId)) {
            $query->where('business_client_id', $businessId);
        }
        $query->orderBy('business_client_id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function deleteDepartment(Request $request) {
        $record = BusinessDepartment::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->delete();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    public function savePeriod(Request $request) {
        $record = new IsrPeriod();
        if ($request->has('id')) {
            $record = IsrPeriod::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        $record->save();

        $lowerLimits = $request->json('lowerLimit');
        $uperLimits = $request->json('uperLimit');
        $fixedFees = $request->json('fixedFee');
        $percents = $request->json('percent');
        for ($i = 0; $i < count($lowerLimits); $i++) {
            $det = new IsrPeriodDetails();
            $det->lower_limit = $lowerLimits[$i];
            $det->uper_limit = $uperLimits[$i];
            $det->fixed_fee = $fixedFees[$i];
            $det->percent = $percents[$i];
            $det->isr_period_id = $record->id;
            $det->save();
        }
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listPeriods(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);

        $query = IsrPeriod::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }

        $query->orderBy('id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function deletePeriod(Request $request) {
        $record = IsrPeriod::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->delete();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    public function savePayroll(Request $request) {
        $record = new Payroll();
        if ($request->has('id')) {
            $record = Payroll::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('importCode')) {
            $record->import_code = $request->json('importCode');
        }
        if ($request->has('rfc')) {
            $record->rfc = $request->json('rfc');
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        if ($request->has('branchId')) {
            $record->business_branch_id = $request->json('branchId');
        }
        if ($request->has('postalCode')) {
            $record->postal_code = $request->json('postalCode');
        }
        if ($request->has('payPeriod')) {
            $record->pay_period = $request->json('payPeriod');
        }
        if ($request->has('businessClient')) {
            $record->business_client_id = $request->json('businessClient');
        }
        if ($request->has('isrPeriod')) {
            $record->isr_period_id = $request->json('isrPeriod');
        }

        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listPayrolls(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $businessId = $request->input('business', null);

        $query = Payroll::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        if (!empty($businessId)) {
            $query->where('business_client_id', $businessId);
        }
        $query->whereNull('canceled_at');
        $query->orderBy('business_client_id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function cancelPayroll(Request $request) {
        $record = Payroll::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->canceled_at = new DateTime();
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    public function savePayrollReceipt(Request $request) {
        $record = new PayrollReceipt();
        if ($request->has('id')) {
            $record = PayrollReceipt::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('code')) {
            $record->import_code = $request->json('code');
        }
        if ($request->has('name')) {
            $record->name = $request->json('name');
        }
        if ($request->has('lastName')) {
            $record->last_name = $request->json('lastName');
        }
        if ($request->has('last_name2')) {
            $record->last_name2 = $request->json('last_name2');
        }
        if ($request->has('rfc')) {
            $record->rfc = $request->json('rfc');
        }
        if ($request->has('curp')) {
            $record->curp = $request->json('curp');
        }
        if ($request->has('workDays')) {
            $record->work_days = $request->json('workDays');
        }
        if ($request->has('subtotal')) {
            $record->subtotal = $request->json('subtotal');
        }
        if ($request->has('isr')) {
            $record->isr = $request->json('isr');
        }
        if ($request->has('total')) {
            $record->total = $request->json('total');
        }
        if ($request->has('payPeriod')) {
            $record->pay_period = $request->json('payPeriod');
        }
        if ($request->has('department')) {
            $record->department = $request->json('department');
        }
        if ($request->has('numEmpleado')) {
            $record->num_empleado = $request->json('numEmpleado');
        }
        if ($request->has('claveEntFed')) {
            $record->clave_ent_fed = $request->json('claveEntFed');
        }

        if ($request->has('log')) {
            $record->log = $request->json('log');
        }
        $record->save();

        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listPayrollReceipts(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $code = $request->input('code', null);
        $rfc = $request->input('rfc', null);
        $payroll = $request->input('payRoll', null);

        $query = PayrollReceipt::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        if (!empty($code)) {
            $query->where('code', $code);
        }
        if (!empty($rfc)) {
            $query->where('rfc', $rfc);
        }
        if (!empty($payroll)) {
            $query->where('payroll_id', $payroll);
        }
        $query->whereNull('canceled_at');
        $query->orderBy('payroll_id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function cancelPayrollReceipt(Request $request) {
        $record = PayrollReceipt::find($request->json('id'));
        if (!$record) {
            return response()->json([
                'status' => 'warning',
                'message' => 'El registro no se encuentra'
            ]);
        }
        $record->canceled_at = new DateTime();
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Map excel information into response data
     */
    function importEmployees(Request $request) {
        if (!$request->has('excel')) {
            return response()->json([
                'status' => 'warning',
                'message' => 'Please provide [excel] to transform'
            ]);
        }

        try {
            $data = $request->json('excel');
            $map = [
                [ 'column' => 'A', 'target' => 'partnership' ],
                [ 'column' => 'B', 'target' => 'middle_name' ],
                [ 'column' => 'C', 'target' => 'last_name'],
                [ 'column' => 'D', 'target' => 'name' ],
                [ 'column' => 'E', 'target' => 'nss' ],
                [ 'column' => 'F', 'target' => 'curp' ],
                [ 'column' => 'G', 'target' => 'rfc' ],
                [ 'column' => 'H', 'target' => 'sdi' ],
                [ 'column' => 'I', 'target' => 'integration_factor'],
                [ 'column' => 'J', 'target' => 'started_at' ],
                [ 'column' => 'K', 'target' => 'factor' ],
                [ 'column' => 'L', 'target' => 'cuota_fija' ],
                [ 'column' => 'M', 'target' => 'prima_riesgo' ],
                [ 'column' => 'N', 'target' => 'workstation_risk' ],
                [ 'column' => 'O', 'target' => 'payment_period' ],
                [ 'column' => 'P', 'target' => 'emissor' ],
                [ 'column' => 'Q', 'target' => 'federative_key' ]
            ];

            $partnerships = Partnership::all();
            $clients = BusinessClient::where('type', 'I')->whereNull('canceled_at')->get();
            $mapper = new MapperController();
            $employees = $mapper->getEmployees(
                $mapper->readExcel($data, $map)
            );
            $records = 0;
            foreach ($employees as $emp) {
                $partnership = null;
                foreach ($partnerships as $b) {
                    if ($b->code === strtoupper($emp['partnership'])) {
                        $partnership = $b;
                    break;
                    }
                }
                if ($partnership) {
                    $empRecord = Employee::where('rfc', $emp['rfc'])
                        ->where('partnership_id', $partnership->id)
                        ->get()->first();
                    if (!$empRecord) {
                        $empRecord = new Employee();
                        $empRecord->partnership_id = $partnership->id;
                        $empRecord->employee_number = CodeService::saveNext(
                            '.' . $partnership->prefix, $partnership->code, 0
                        );
                        $empRecord->name = $emp['name'];
                        $empRecord->last_name = $emp['middle_name'];
                        $empRecord->last_name2 = $emp['last_name'];
                        $empRecord->rfc = strtoupper($emp['rfc']);
                        $empRecord->curp = $emp['curp'];
                    } else if ($emp['partnership'] != $empRecord->partnership->code) {
                        $code = $emp['partnership'];
                         Log::info("Empleado: $empRecord->full_name <> $code <> $empRecord->partnerhip->code");
                    }
                    $empRecord->type = 'W';
                    $empRecord->nss = $emp['nss'];
                    $empRecord->sdi = $emp['sdi'];               
                    $empRecord->integration_factor = $emp['integration_factor'];
                    $date = Carbon::createFromFormat('Y-m-d', $emp['started_at']); //date("d-m-Y", strtotime($emp['started_at']));  //date_create_from_format('Y-m-d', $emp['started_at']);
                    $empRecord->started_at = $date;
                    if (!!$emp['factor']) {
                        $empRecord->infonavit_type = 'F';
                        $empRecord->infonavit_value = $emp['factor'];
                    } else if (!!$emp['cuota_fija']) {
                        $empRecord->infonavit_value = $emp['cuota_fija'];
                        $empRecord->infonavit_type = 'C';
                    } else {
                        $empRecord->infonavit_value = null;
                        $empRecord->infonavit_value = null;
                    }
                    
                    $empRecord->workstation_risk = $emp['workstation_risk'];
                    $empRecord->payment_period = $emp['payment_period'];
                    foreach ($clients as $c) {
                        if ($c->code === strtoupper($emp['emissor'])) {
                            $empRecord->business_client_id = $c->id;
                        break;
                        }
                    }
                    $empRecord->federative_key = $emp['federative_key'];

                    $empRecord->status = 'R';
                    $empRecord->payroll_type = 'O';
                    $empRecord->regime_type = '02';
                    $empRecord->save();
                    $records ++;
                }
            }
            return response()->json([
                'status' => 'ok',
                'records' => $records
            ]);
        } catch(Exception $ex) {
            report($ex);
            return response()->json([
                'status' => 'error',
                'message' => $ex->getMessage()
            ]);
        }
    }

    /**
     * Map excel information into response data
     */
    function importEmployeeAccounts(Request $request) {
        if (!$request->has('excel')) {
            return response()->json([
                'status' => 'warning',
                'message' => 'Please provide [excel] to transform'
            ]);
        }

        try {
            $data = $request->json('excel');
            $map = [
                [ 'column' => 'A', 'target' => 'partnership' ],
                [ 'column' => 'G', 'target' => 'rfc'  ],
                [ 'column' => 'H', 'target' => 'account' ],
                [ 'column' => 'I', 'target' => 'bank' ]
            ];

            $banks = Bank::all();
            $partnerships = Partnership::all();
            $mapper = new MapperController();
            $employees = $mapper->getAccountEmployees(
                $mapper->readExcel($data, $map)
            );
            $records = 0;
            foreach ($employees as $emp) {
                $partnership = null;
                foreach ($partnerships as $b) {
                    if ($b->code === strtoupper($emp['partnership'])) {
                        $partnership = $b;
                    }
                }

                if ($partnership) {
                    $empRecord = Employee::where('rfc', $emp['rfc'])
                        ->where('partnership_id', $partnership->id)
                        ->get()->first();
                    if ($empRecord != null) {
                        $bank = null;
                        foreach ($banks as $b) {
                            if ($b->name == $emp['bank']) {
                                $bank = $b;
                                break;
                            }
                        }
                        $empRecord->account = $emp['account'];
                        if ($bank != null) {
                            $empRecord->bank()->associate($bank);
                        }
                        $empRecord->save();
                        $records ++;
                    }
                }
            }
            return response()->json([
                'status' => 'ok',
                'records' => $records
            ]);
        } catch(Exception $ex) {
            report($ex);
            return response()->json([
                'status' => 'error',
                'message' => $ex->getMessage()
            ]);
        }
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listEmployees(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $rfc = $request->input('rfc', null);
        $businessId = $request->input('businessId', null);

        $query = Employee::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        if (!empty($rfc)) {
            $query->where('rfc', $rfc);
        }
        if (!empty($businessId)) {
            $query->where('business_client_id', $businessId);
        }
        $query->whereNull('canceled_at');
        $query->orderBy('id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    /**
     * Muestra log
     */
    public function showLog(Request $request, $fileName) {
        //dd(Storage::disk('logs')->get('laravel-2019-09-30.log'));
        dd(Storage::disk('logs')->get('laravel-' . $fileName . '.log'));
    }

    public function countBills(Request $request, $strFrom, $strTo) {
        $fromDate = Carbon::parse($strFrom);
        $toDate = Carbon::parse($strTo . ' 23:59:59');

        $invoices = Invoice::select('id')->where('stamp_signer', 'SI')
            ->where('stamped_at', '>=', $fromDate)
            ->where('stamped_at', '<=', $toDate)
            ->count();
        $cancelInvoices = Invoice::select('id')->where('cancel_signer', 'SI')
            ->where('canceled_at', '>=', $fromDate)
            ->where('canceled_at', '<=', $toDate)
            ->count();
        
        $assimilated = PayrollReceipt::select('id')->where('stamp_signer', 'SI')
            ->where('stamped_at', '>=', $fromDate)
            ->where('stamped_at', '<=', $toDate)
            ->count();
        $cancelAssimilated = PayrollReceipt::select('id')->where('cancel_signer', 'SI')
            ->where('canceled_at', '>=', $fromDate)
            ->where('canceled_at', '<=', $toDate)
            ->count();
        
        $workers = PayrollWorkerReceipt::select('id')->where('stamp_signer', 'SI')
            ->where('stamped_at', '>=', $fromDate)
            ->where('stamped_at', '<=', $toDate)
            ->count();
        $cancelWorkers = PayrollWorkerReceipt::select('id')->where('cancel_signer', 'SI')
            ->where('canceled_at', '>=', $fromDate)
            ->where('canceled_at', '<=', $toDate)
            ->count();
        $totals = $invoices + $cancelInvoices + $assimilated + $cancelAssimilated + $workers + $cancelWorkers;

        return response()->json([
            'From' => $fromDate->format('Y-m-d H:m'),
            'To' => $toDate->format('Y-m-d H:m'),
            'Invoices' => $invoices,
            'Cancel Invoices' => $cancelInvoices,
            'Assimilated' => $assimilated,
            'Cancel Assimilated' => $cancelAssimilated,
            'Workers' => $workers,
            'Cancel Workers' => $cancelWorkers,
            'Totals' => $totals
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listConfigCodes(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);
        $query = ConfigCodes::skip($page * $pageSize)->take($pageSize);
        if (!empty($filter)) {
            $query->where('name', 'like', '%' . $filter . '%');
        }
        $query->orderBy('id', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }

    public function saveConfigCode(Request $request) {
        $record = new ConfigCodes();
        if ($request->has('id')) {
            $record = ConfigCodes::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('type')) {
            $record->type = $request->json('type');
        }
        if ($request->has('pattern')) {
            $record->pattern = $request->json('pattern');
        }
        if ($request->has('length')) {
            $record->length = $request->json('length');
        }
        if ($request->has('value')) {
            $record->value = $request->json('value');
        }
        
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Save record
     * @param request Request data
     * @param category category to load
     */
    public function saveProcessRequest(Request $request) {
        $record = new ProcessRequest();
        if ($request->has('id')) {
            $record = ProcessRequest::find($request->json('id'));
            if (!$record) {
                return response()->json([
                    'status' => 'warning',
                    'message' => 'El registro no se encuentra'
                ]);
            }
        }
        if ($request->has('code')) {
            $record->code = $request->json('code');
        }
        if ($request->has('status')) {
            $record->status = $request->json('status');
        }
        if ($request->has('command')) {
            $record->command = $request->json('command');
        }
        if ($request->has('progress')) {
            $record->progress = $request->json('progress');
        }
        if ($request->has('expiredAt')) {
            $record->expired_at = $request->json('expiredAt');
        }
        if ($request->has('description')) {
            $record->description = $request->json('description');
        }
        if ($request->has('linkCode')) {
            $record->link_code = $request->json('linkCode');
        }
        $record->save();
        return response()->json([
            'status' => 'ok',
            'data' => $record->format()
        ]);
    }

    /**
     * Retrieve list data
     * @param request Request data
     */
    public function listProcessRequest(Request $request) {
        $pageSize = $request->input('pageSize', 25);
        $page = $request->input('page', 0);
        $filter = $request->input('filter', null);

        $query = ProcessRequest::skip($page * $pageSize)->take($pageSize);
        if ($request->has('id')) {
            $query->where('id', $request->input('id'));
        } else if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('code', 'like', '%' . $filter . '%')
                    ->orWhere('command', 'like', '%' . $filter . '%')
                    ->orWhere('link_code', 'like', '%', $filter . '%');
            });
        }
        $query->orderBy('created_at', 'desc');
        $result = array();
        foreach($query->get() as $record) {
            array_push($result, $record->format());
        }

        return response()->json([
            'status' => 'ok',
            'data' => $result
        ]);
    }
}
