<?php

namespace App\Http\Controllers\InternalClient;

use App\Http\Controllers\Controller;
use App\Model\BusinessClient;
use App\Model\BusinessClientConcept;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class BusinessClientController extends Controller
{
    
    /**
     * Retrieves a list data of internal business client
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at');
        if ($status != 'V') {
            $query = BusinessClient::whereNotNull('canceled_at');
        }
        
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('code', 'like', '%' . $filter . '%')
                    ->orWhere('name', 'like', '%' . $filter . '%')
                    ->orWhere('rfc', 'like', '%' . $filter . '%');
            });
        }
        
        $total = $query->count('id');
        $query->orderBy('name')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        return view('admin.business-client-index', [
            'businessClients' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'status' => $status
        ]);
    }

    /**
     * Shows a business client form for add a business client
     */
    public function create(Request $request) {
        $business = new BusinessClient();
        $departments = [];
        $clientConcepts = [];
        $specialConcepts = [];
        return view('admin.business-client', [
            'business' => $business,
            'departments' => $departments,
            'clientConcepts' => $clientConcepts,
            'specialConcepts' => $specialConcepts
        ]);
    }

    /**
     * Shows a business client form for edit his data
     * 
     * @param request request data
     * @param id The id of business client
     */
    public function edit(Request $request, $id) {
        $business = BusinessClient::find($id);
        $clientConcepts = BusinessClientConcept::
            where('business_client_id', $business->id)
            ->where('type', 'C')
            ->get();
        $specialConcepts = BusinessClientConcept::
            where('business_client_id', $business->id)
            ->where('type', 'S')
            ->get();
        return view('admin.business-client', [
            'business' => $business,
            'clientConcepts' => $clientConcepts,
            'specialConcepts' => $specialConcepts
        ]);
    }

    /**
     * Stores a business cliente, filled on form request
     * 
     * @param request The request data with values
     */
    public function store(Request $request) {
        $this->validBusinessClientForm($request);

        $business = new BusinessClient();
        if ($request['id'] != null) {
            $business = BusinessClient::find($request['id']);
        }
        $business->code = strtoupper($request['code']);
        $business->name = $request['name'];
        $business->employer_registration = $request['employer_registration'];
        $business->rfc = strtoupper($request['rfc']);
        $business->prefix = strtoupper($request['prefix']);
        $business->image = $request['image'];
        $business->type = 'I';
        $business->save();

        Cache::put('message', 'Se salvo correctamente al cliente interno ' . 
            $business->name);
        return redirect('/admin/internal-business-clients/' . $business->id . '/edit');
    }

    /**
     * Remove a business client from specified business
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un cliente interno');
        } else {
            BusinessClient::whereIn('id', $request->input('selected'))
                ->where('type', 'I')
                ->update(['canceled_at' => Carbon::now()]);
            Cache::put('message', 'Los clientes internos han sido dados de baja');
        }
        return redirect('/admin/internal-business-clients');
    }

    /**
     * Valid request form
     */
    public function validBusinessClientForm(Request $request) {
        $request->validate([
            'code' => 'required',
            'name' => 'required',
            'rfc' => 'required',
            'prefix' => 'required'
        ], [
            'code.required' => 'El código es requerido',
            'name.required' => 'El nombre es requerido',
            'rfc.required' => 'El rfc es requerido',
            'prefix.required' => 'El prefijo es requerido'
        ]);
    }
}
