<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Model\IsrPeriod;
use App\Model\IsrPeriodDetails;
use App\Model\SubsidyDetail;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class ISRPeriodController extends Controller
{
    /**
     * Execute command
     * @param request Controller request
     */
    public function command(Request $request) {
        if ($request->input('action') === 'search') {
            $request->merge(['page-index' => 0]);
            return $this->index($request);
        } else if ($request->input('action') === 'search-page') {
            return $this->index($request);
        } else if ($request->input('action') === 'destroy') {
            return $this->destroy($request);
        } else if ($request->input('action') === 'export') {
            return $this->export($request);
        } else if ($request->input('action') === 'store') {
            return $this->store($request);
        } else if ($request->input('action') === 'create') {
            return $this->create($request);
        }
        return redirect()->back()->withInput($request->input());
    }
    
    /**
     * Creates a query throught Request inputs
     */
    public function getQuery(Request $request) {
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = IsrPeriod::select('*'); 
        if ($status == 'V') {
            $query = IsrPeriod::whereNull('vigented_at');
        } else {
            $query = IsrPeriod::whereNotNull('vigented_at');
        }
        
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('name', 'like', $filter . '%');
            });
        }
        return $query;
    }

    /**
     * Retrieves a list of employees
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = $this->getQuery($request);
        $total = $query->count('id');
        
        $query->orderBy('name')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('admin.isr-period-index', [
            'periods' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'status' => $status
        ]);
    }

    /**
     * Creates a form for add a new period
     */
    public function create(Request $request) {
        $period = new IsrPeriod();
        $details = [];
        $subsidies = [];
        for ($i = 0; $i < 11; $i ++) {
            array_push($details, new IsrPeriodDetails());
            array_push($subsidies, new SubsidyDetail());
        }
        return view('admin.isr-period', [
            'period' => $period,
            'details' => $details,
            'subsidies' => $subsidies
        ]);
    }

    /**
     * Save a isr period
     */
    public function store(Request $request) {
        $id = $request->input('id', 0);
        $query = IsrPeriod::whereNull('vigented_at');
        if ($id != 0) {
            $query->where('id', '<>', $id);
        }

        $period = new IsrPeriod();
        if ($request['id'] != null) {
            $period = IsrPeriod::find($request['id']);
        }
        $period->name = $request->input('name', '');
        $period->days = $request->input('days', 1);
        $period->save();
        
        if ($request->input('detail-id')) {
            for ($i = 0; $i < 11; $i++) {
                $detail = new IsrPeriodDetails();
                $detail->isrPeriod()->associate($period);
                if ($request->input('detail-id')[$i]) {
                    $detail = IsrPeriodDetails::find($request->input('detail-id')[$i]);
                }
                $detail->lower_limit = $request->input('detail-lower-limit')[$i];
                $detail->uper_limit = $request->input('detail-uper-limit')[$i];
                $detail->fixed_fee = $request->input('detail-fixed-fee')[$i];
                $detail->percent = $request->input('detail-percent')[$i];
                $detail->save();
            }

            for ($i = 0; $i < 11; $i++) {
                $detail = new SubsidyDetail();
                $detail->isrPeriod()->associate($period);
                if ($request->input('subsidy-id')[$i]) {
                    $detail = SubsidyDetail::find($request->input('subsidy-id')[$i]);
                }
                $detail->lower_limit = $request->input('subsidy-lower-limit')[$i];
                $detail->uper_limit = $request->input('subsidy-uper-limit')[$i];
                $detail->subsidy = $request->input('subsidy-percent')[$i];
                $detail->save();
            }
        }

        Cache::put('message', 'Se salvo correctamente al periodo' . 
            $period->name);
        return redirect('/admin/isr-periods/' . $period->id . '/edit');
    }

    /**
     * Shows a employee form for edit his data
     */
    public function edit(Request $request, $id) {
        $period = IsrPeriod::find($id);
        $details = $period->details;
        $subsidies = $period->subsidyDetails;
        if (count($subsidies) == 0) {
            $subsidies = [];
            for ($i = 0; $i < 11; $i ++) {
                $detail = new SubsidyDetail();
                array_push($subsidies, $detail);
            }
        }

        return view('admin.isr-period', [
            'period' => $period,
            'details' => $details,
            'subsidies' => $subsidies
        ]);
    }

    /**
     * Removes a period
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un periodo');
        } else {
            IsrPeriod::whereIn('id', $request->input('selected'))
                ->update(['vigented_at' => Carbon::now()]);
            Cache::put('message', 'Los periodos han sido dados de baja');
        }
        return redirect('/admin/isr-periods');
    }    
}
