<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Exports\EmployeeExports;
use App\Http\Services\CatCFDIService;
use App\Http\Services\CodeService;
use App\Model\Bank;
use App\Model\BusinessClient;
use App\Model\BusinessClientEmployee;
use App\Model\Employee;
use App\Model\Partnership;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class EmployeeController extends Controller
{

    /**
     * Execute command
     * @param request Controller request
     */
    public function command(Request $request) {
        if ($request->input('action') === 'search') {
            $request->merge(['page-index' => 0]);
            return $this->index($request);
        } else if ($request->input('action') === 'search-page') {
            return $this->index($request);
        } else if ($request->input('action') === 'destroy') {
            return $this->destroy($request);
        } else if ($request->input('action') === 'export') {
            return $this->export($request);
        } else if ($request->input('action') === 'store') {
            return $this->store($request);
        } else if ($request->input('action') === 'create') {
            return $this->create($request);
        }
        return redirect()->back()->withInput($request->input());
    }
    
    /**
     * Creates a query throught Request inputs
     */
    public function getQuery(Request $request) {
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = Employee::select('employees.*')
            ->join('partnerships', 'partnerships.id', '=', 'employees.partnership_id')
            ->where('employees.status', 'R');
        if ($status == 'V') {
            $query->whereNull('employees.canceled_at');
        } else {
            $query->whereNotNull('employees.canceled_at');
        }
        
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where(DB::raw('concat(employees.name, " ", ' . 'employees.last_name, " ", employees.last_name2)'), 
                    'like', '%' . $filter . '%')
                    ->orWhere('employees.rfc', 'like', '%' . $filter . '%')
                    ->orWhere('employees.employee_number', 'like', '%' . $filter . '%');
            });
        }
        $partnershipId = $request->input('partnership', 'T');
        if ($partnershipId != 'T') {
            $query->where('employees.partnership_id', $partnershipId);
        }
        return $query;
    }

    /**
     * Retrieves a list of employees
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = $this->getQuery($request);
        
        $total = $query->count('employees.id');
        
        $query->orderBy('employees.partnership_id')
            ->orderBy(DB::raw('cast(employees.employee_number as unsigned)'))
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('admin.employee-index', [
            'employees' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'partnerships' => Partnership::withTrashed()->orderBy('code')->get(),
            'partnershipId' => $request->input('partnership'),
            'status' => $status
        ]);
    }

    /**
     * Creates a form for add a new employee
     */
    public function create(Request $request) {
        $employee = new Employee();
        return view('admin.employee', [
            'employee' => $employee,
            'partnerships' => Partnership::all()
        ]);
    }

    /**
     * Save a employee
     */
    public function store(Request $request) {
        $id = $request->input('id', 0);

        $rfc = strtoupper($request->input('rfc'));
        $query = Employee::whereNull('canceled_at')
            ->where('status', 'R')
            ->where('rfc', $rfc);
        if ($id != 0) {
            $query->where('id', '<>', $id);
        }
        if ($query->count() > 0) {
            Cache::put('warning', 'El rfc ' . $rfc . ' ya esta relacionado a otro empleado');            
            return redirect()->back()->withInput($request->input());
        }
        $employee = new Employee();
        if ($request['id'] != null) {
            $employee = Employee::find($request['id']);
        }
        $employee->name = $request->input('name', '');
        $employee->last_name = $request->input('last-name', '');
        $employee->last_name2 = $request->input('last-name2', '');
        $employee->rfc = $rfc;
        $employee->curp = strtoupper($request->input('curp'));
        $employee->type = $request->input('type');
        $employee->partnership_id = $request->input('partnership', null);        
        if (!$employee->employee_number) {
            $partnership = $employee->partnership;
            $employee->employee_number = CodeService::saveNext(
                '.' . $partnership->prefix, $partnership->code, 0
            );
        }
        $employee->status = 'R';
        $employee->save();
        
        foreach ($employee->BusinessClientEmployees as $clientEmployee) {
            $exist = false;
            if ($request->input('business')) {
                for ($i = 0; $i < count($request->input('business')); $i++) {
                    if ($clientEmployee->business_client_id == $request->input('business')[$i]) {
                        $exist = true;
                        break;
                    }
                }
            }
            if (!$exist) {
                $clientEmployee->delete();
            }
        }
        if ($request->input('business')) {
            for ($i = 0; $i < count($request->input('business')); $i++) {
                if (!$employee->isActive($request->input('business')[$i])) {
                    $businessEmployee = new BusinessClientEmployee();
                    $businessEmployee->employee()->associate($employee);
                    $businessEmployee->business_client_id = $request->input('business')[$i];
                    $businessEmployee->save();
                }
            }
        }

        Cache::put('message', 'Se salvo correctamente al empleado ' . $employee->getFullName());
        return redirect('/admin/employees/' . $employee->id . '/edit');
    }

    public function storeEmployee(Request $request, $id) {
        $employee = Employee::find($id);

        $query = Employee::whereNull('canceled_at')
            ->where('partnership_id', $employee->partnership_id)
            ->where('rfc', strtoupper($request->input('rfc')));
        if ($id != 0) {
            $query->where('id', '<>', $id);
        }
        if ($query->count() > 0) {
            Cache::put('warning', 'El rfc ' . strtoupper($request->input('rfc', '')) 
                . ' ya esta relacionado a otro empleado');
            return redirect()->back()->withInput($request->input());
        }

        if ($request->input('started-at')) {
            $date = date_create_from_format('Y-m-d', $request->input('started-at'));
            $employee->started_at = $date;
        }
        if ($request->input('client')) {
            $employee->business_client_id = $request->input('client');
        }
        $employee->type = $request->input('type');
        $employee->name = $request->input('name');
        $employee->last_name = $request->input('last-name');
        $employee->last_name2 = $request->input('last-name2');
        $employee->rfc = strtoupper($request->input('rfc'));
        $employee->curp = strtoupper($request->input('curp'));
        $employee->address = $request->input('address');
        $employee->civil_status = $request->input('civil-status');
        $employee->bank_id = $request->input('bank');
        $employee->account = $request->input('account');
        //$employee->card_number = $request->input('card-number');
        $employee->sdi = $request->input('sdi');
        $employee->salary = $request->input('salary');
        $employee->integration_factor = $request->input('integration-factor');

        if ($request->input('infonavit-type') == '0') {
            $employee->infonavit_type = null;
            $employee->infonavit_value = null;
        } else {
            $employee->infonavit_type = $request->input('infonavit-type');
            $employee->infonavit_value = $request->input('infonavit-value');
        }

        $employee->federative_key = $request->input('federative-key');
        $employee->payment_period = $request->input('payment-period');
        $employee->payroll_type = $request->input('payroll-type');
        $employee->workstation_risk = $request->input('workstation-risk');
        $employee->regime_type = $request->input('regime-type');
        $employee->contract_type = $request->input('contract-type');
        $employee->employer_register = $request->input('employer-register');

        if ($request->has('nss')) {
            $employee->nss = $request->input('nss');
        }
        if ($request->has('workstation')) {
            $employee->workstation = $request->input('workstation');
        }
        $employee->status = 'R';
        $employee->save();

        if ($request->hasFile('ine-file')) {
            $file = $request->file('ine-file');
            $employee->ine_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/ine/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->ine_file, $fileContent);
        }
        if ($request->hasFile('address-file')) {
            $file = $request->file('address-file');
            $employee->address_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/address/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->address_file, $fileContent);
        }
        if ($request->hasFile('rfc-file')) {
            $file = $request->file('rfc-file');
            $employee->rfc_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/rfc/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->rfc_file, $fileContent);
        }
        if ($request->hasFile('curp-file')) {
            $file = $request->file('curp-file');
            $employee->curp_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/curp/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->curp_file, $fileContent);
        }
        if ($request->hasFile('infonavit-file')) {
            $file = $request->file('infonavit-file');
            $employee->infonavit_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/infonavit/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->infonavit_file, $fileContent);
        }
        if ($request->hasFile('nss-file')) {
            $file = $request->file('nss-file');
            $employee->nss_file = 'partnerships' . '/' . $employee->partnership->code . '/employees/' . $employee->id
                . '/nss/' . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->nss_file, $fileContent);
        }

        foreach ($employee->BusinessClientEmployees as $clientEmployee) {
            $exist = false;
            if ($request->input('business')) {
                for ($i = 0; $i < count($request->input('business')); $i++) {
                    if ($clientEmployee->business_client_id == $request->input('business')[$i]) {
                        $exist = true;
                        break;
                    }
                }
            }
            if (!$exist) {
                $clientEmployee->delete();
            }
        }
        if ($request->input('business')) {
            for ($i = 0; $i < count($request->input('business')); $i++) {
                if (!$employee->isActive($request->input('business')[$i])) {
                    $businessEmployee = new BusinessClientEmployee();
                    $businessEmployee->employee()->associate($employee);
                    $businessEmployee->business_client_id = $request->input('business')[$i];
                    $businessEmployee->save();
                }
            }
        }
        Cache::put('message', 'Se salvo correctamente al empleado ' . $employee->getFullName());
        return redirect('/admin/employees/' . $employee->id . '/edit');
    }

    public function changePartnership(Request $request, $id) {
        $employee = Employee::find($id);

        if ($request->input('started-at')) {
            $date = date_create_from_format('Y-m-d', $request->input('started-at'));
            $employee->started_at = $date;
        }
        if ($employee->partnership_id != $request->input('partnership')) {
            foreach ($employee->businessClients as $client) {
                $client->delete();
            }
            $employee->partnership_id = $request->input('partnership');
            $employee->save();
        }
        Cache::put('message', 'Se cambio correctamente al consorcio ' . $employee->getFullName());
        return redirect('/admin/employees/' . $employee->id . '/edit');
    }

    /**
     * Shows a employee form for edit his data
     */
    public function edit(Request $request, $id) {
        $employee = Employee::find($id);
        $clients = BusinessClient::where('type', 'I')->whereNull('canceled_at')->get();

        $banks = Bank::all();
        return view('admin.edit-employee', [
            'record' => $employee,
            'banks' => $banks,
            'editable' => true,
            'adminFields' => 1,
            'clients' => $clients,
            'partnerships' => Partnership::orderBy('code')->get(),
            'federativeKeys' => CatCFDIService::getFederativeKeys(),
            'payrollTypes' => CatCFDIService::getPayrollTypes(),
            'contractTypes' => CatCFDIService::getContractTypes(),
            'regimeTypes' => CatCFDIService::getRegimeTypes(),
            'workstationRisks' => CatCFDIService::getWorkstationRisks(),
            'paymentPeriods' => CatCFDIService::getPaymentPeriod()
        ]);
    }

    /**
     * Remove a employee from specified business
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un empleado');
        } else {
            Employee::whereIn('id', $request->input('selected'))
                ->update(['canceled_at' => Carbon::now()]);
            Cache::put('message', 'Los empleados han sido dados de baja');
        }
        return redirect('/admin/employees');
    }

    /**
     * Export employee data of specified client
     * @param request Controller request
     */
    public function export(Request $request) {
        $export = new EmployeeExports();
        $export->query = $this->getQuery($request);

        $partnershipId = $request->input('partnership', 'T');
        if ($partnershipId == 'T') {
            $export->query->whereNull('partnerships.deleted_at');
        }
        $export->query->select('employees.*');
        $export->query->with('bank');

        return $export->download('empleados.xlsx');
    }

    /**
     * Export employee data of specified external client
     * @param request Controller request
     */
    public function exportFromExternalBusiness(Request $request, $id) {
        $export = new EmployeeExports();
        $query = Employee::join('business_client_employees',
            'business_client_employees.employee_id', '=', 'employees.id')
            ->where('business_client_employees.business_client_id', $id)
            ->whereNull('employees.canceled_at')
            ->whereNull('business_client_employees.deleted_at')
            ->select('employees.*');
            
        $export->query = $query;
        return $export->download('empleados.xlsx');
    }
}
