<?php

namespace App\Http\Controllers;

use App\Http\Services\CodeService;
use App\Http\Services\DocumentServices;
use App\Http\Services\InvoiceService;
use App\Model\Invoice;
use App\Model\PayrollReceipt;
use App\Model\PayrollWorkerReceipt;
use BaconQrCode\Encoder\QrCode;
use DOMDocument;
use PDF;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use NumerosEnLetras;
use Idevman\XmlMapper\Support\Facades\XmlMapper;
use Illuminate\Support\Facades\Log;
use SimpleSoftwareIO\QrCode\Facades\QrCode as FacadesQrCode;

class DownloadController extends Controller
{

    private function getStringNumber($total) {
        $totalInteger = floor($total);
        $totalDecimal = round($total - $totalInteger, 2) * 100;
        $totalString = strtoupper(NumerosEnLetras::convertir($totalInteger)) . ' PESOS';
        if ($totalDecimal > 0) {
            $totalString .= ' CON ' . strtoupper(NumerosEnLetras::convertir($totalDecimal)) . ' CENTAVOS';
        }
        return $totalString;
    }
    /**
     * Allow do download a resource
     */
    public function download(Request $request) {
        $mime = $request->get('kind') === 'xml' ?
            'text/xml' : 'application/pdf';
        return response(
            Storage::disk('local')->get($request->get('file')
                + '.' + $request->get('kind')),
             200, [ 'Content-Type' => $mime ]);
    }

    /**
     * Download requested XML
     */
    public function downloadXML(Request $request, $id) {
        $payroll = PayrollReceipt::find($id);
        $content = $this->getContentXML($payroll);
        return response($content)
            ->header('Content-Type', 'text/xml')
            ->header('Content-Disposition', 'attachment; filename="' . $payroll->getFullName() . '.xml"');
    }

    public function getContentXML($receipt) {
        if ($receipt->uuid) {
            return Storage::disk('s3')->get($receipt->getStampPath() . $receipt->uuid . '.xml');
        }
        return DocumentServices::buildXMLContent($receipt);
    }

    /**
     * Download requested XML
     */
    public function downloadSysXML(Request $request, $id) {
        $payroll = PayrollWorkerReceipt::find($id);
        $content = $this->getSysContentXML($payroll);
        return response($content)
            ->header('Content-Type', 'text/xml')
            ->header('Content-Disposition', 'attachment; filename="' . $payroll->full_name . '.xml"');
    }

    public function getSysContentXML($receipt) {
        if ($receipt->uuid) {
            return Storage::disk('s3')->get($receipt->xml_file);
        }
        return DocumentServices::buildSysXMLContent($receipt);
    }

    /**
     * Download requested PDF
     */
    public function downloadSysPDF(Request $request, $id) {
        $payroll = PayrollWorkerReceipt::find($id);
        return $this->getContentSysPDF($payroll);
    }

    /**
     * Download requested PDF
     */
    public function getContentSysPDF($receipt) {
        $qrcode = null;
        if ($receipt->isStamped()) {
            $xml = Storage::disk('s3')->get($receipt->xml_file);
            $map = XmlMapper::mapTo([
                    'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]',
                    'timbreFiscal[NoCertificadoSAT,SelloSAT,SelloCFD]' => '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[][@NoCertificadoSAT,@SelloSAT,@SelloCFD]'
                ], $xml);
            //$qrcode = base64_encode(Storage::disk('s3')->get(
            //    $payroll->getStampPath() . $payroll->uuid . '.png'));
            $qr = 'https://verificacfdi.facturaelectronica.sat.gob.mx/default.aspx?id=' 
                . $receipt->uuid . '&re=' . $receipt->employee->businessClient->rfc 
                . '&rr=' . $receipt->rfc 
                . '&tt=' . number_format($receipt->total, 6)
                . '&fe=' . substr($map['timbreFiscal'][0]['SelloCFD'], -8);
            $qrcode = base64_encode(
                FacadesQrCode::format('png')->size(200)->generate($qr)
            );
        } else {
            $xml = DocumentServices::buildSysXMLContent($receipt);
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]'
            ], $xml);
        }
        
        $totalString = '( ' . $this->getStringNumber(round($receipt->total, 2)) . ' M.N. )';
        $stamp = [
            'noCertificado' => $map['comprobante'][0]['NoCertificado'],
            'certificadoSAT' => isset($map['timbreFiscal']) ? 
                $map['timbreFiscal'][0]['NoCertificadoSAT'] : '',
            'sello' => $map['comprobante'][0]['Sello'],
            'selloSAT' => isset($map['timbreFiscal']) ?
                $map['timbreFiscal'][0]['SelloSAT'] : '',
            'qrcode' => $qrcode,
            'originalString' => InvoiceService::getOriginalString($xml)
        ];
        $businessClient = $receipt->employee->businessClient;
        $pdf = PDF::loadView('formats.worker-format', [
            'payroll' => $receipt,
            'businessClient' => $businessClient,
            'totalString' => $totalString,
            'stamp' => $stamp]);
        if (explode('.', PHP_VERSION)[1] >= 4) {
            error_reporting(E_ALL ^ E_DEPRECATED);
        }
        return response($pdf->output())
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="' . $receipt->full_name . '.pdf"');
    }

    /**
     * Download requested PDF
     */
    public function downloadPDF(Request $request, $id) {
        $payroll = PayrollReceipt::find($id);
        return $this->getContentPDF($payroll);
    }

    /**
     * Download requested PDF
     */
    public function getContentPDF($payroll) {
        $qrcode = null;
        if ($payroll->isStamped()) {
            $xml = Storage::disk('s3')->get(
                $payroll->getStampPath() . $payroll->uuid . '.xml');
            $map = XmlMapper::mapTo([
                    'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]',
                    'timbreFiscal[NoCertificadoSAT,SelloSAT,SelloCFD]' => '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[][@NoCertificadoSAT,@SelloSAT,@SelloCFD]'
                ], $xml);
            //$qrcode = base64_encode(Storage::disk('s3')->get(
            //    $payroll->getStampPath() . $payroll->uuid . '.png'));
            $qr = 'https://verificacfdi.facturaelectronica.sat.gob.mx/default.aspx?id=' 
                . $payroll->uuid . '&re=' . $payroll->payroll->businessClient->rfc 
                . '&rr=' . $payroll->rfc 
                . '&tt=' . number_format($payroll->total, 6)
                . '&fe=' . substr($map['timbreFiscal'][0]['SelloCFD'], -8);
            $qrcode = base64_encode(
                FacadesQrCode::format('png')->size(200)->generate($qr)
            );
        } else {
            $xml = DocumentServices::buildXMLContent($payroll);
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]'
            ], $xml);
        }
        
        $totalString = '( ' . $this->getStringNumber(round($payroll->total, 2)) . ' M.N. )';
        $stamp = [
            'noCertificado' => $map['comprobante'][0]['NoCertificado'],
            'certificadoSAT' => isset($map['timbreFiscal']) ? 
                $map['timbreFiscal'][0]['NoCertificadoSAT'] : '',
            'sello' => $map['comprobante'][0]['Sello'],
            'selloSAT' => isset($map['timbreFiscal']) ?
                $map['timbreFiscal'][0]['SelloSAT'] : '',
            'qrcode' => $qrcode,
            'originalString' => InvoiceService::getOriginalString($xml)
        ];
        $businessClient = $payroll->payroll->businessClient;
        $pdf = PDF::loadView('formats.assimilated-format', [
            'payroll' => $payroll,
            'businessClient' => $businessClient,
            'totalString' => $totalString,
            'stamp' => $stamp]);
        if (explode('.', PHP_VERSION)[1] >= 4) {
            error_reporting(E_ALL ^ E_DEPRECATED);
        }
        return response($pdf->output())
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="' . $payroll->getFullName() . '.pdf"');
    }

    /**
     * Download requested XML
     */
    public function downloadInvoiceXML(Request $request, $id) {
        $invoice = Invoice::find($id);
        $content = $this->getInvoiceContentXML($invoice);
        return response($content)
            ->header('Content-Type', 'text/xml')
            ->header('Content-Disposition', 'attachment; filename="' . $invoice->receiver->name . '.xml"');
    }

    public function getInvoiceContentXML($invoice) {        
        if ($invoice->isStamped()) {
            return Storage::disk('s3')->get($invoice->getStampPath() . $invoice->uuid . '.xml');
        }
        return DocumentServices::buildInvoiceXMLContent($invoice);
    }

    /**
     * Download requested PDF
     */
    public function downloadInvoicePDF(Request $request, $id) {
        $invoice = Invoice::find($id);
        if (!$invoice->isStamped()) {
            $invoice->code = CodeService::getNext(
                $invoice->emissor->getCodePattern($invoice->business_branch_id),
                $invoice->emissor->getTypeForCodes()
            );
        }
        $qrcode = null;
        if ($invoice->isStamped()) {
            $xml = Storage::disk('s3')->get(
                $invoice->getStampPath() . $invoice->uuid . '.xml');
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => 
                    '/cfdi:Comprobante[][@NoCertificado,@Sello]',
                'timbreFiscal[NoCertificadoSAT,SelloSAT,SelloCFD]' => 
                    '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[][@NoCertificadoSAT,@SelloSAT,@SelloCFD]'
                ], $xml);
            /*
            $qrcode = base64_encode(Storage::disk('s3')->get(
                $invoice->getStampPath() . $invoice->uuid . '.png'));
            */
            $qr = 'https://verificacfdi.facturaelectronica.sat.gob.mx/default.aspx?id=' 
                . $invoice->uuid . '&re=' . $invoice->emissor->rfc 
                . '&rr=' . $invoice->receiver->rfc 
                . '&tt=' . number_format($invoice->total, 6)
                . '&fe=' . substr($map['timbreFiscal'][0]['SelloCFD'], -8);
            $qrcode = base64_encode(
                FacadesQrCode::format('png')->size(200)->generate($qr)
            );
        } else {
            $xml = DocumentServices::buildInvoiceXMLContent($invoice);
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => 
                    '/cfdi:Comprobante[][@NoCertificado,@Sello]'
            ], $xml);
        }

        $totalString = strtoupper($this->getStringNumber($invoice->calcTotalNeto())) . ' M.N.';
        $stamp = [
            'noCertificado' => $map['comprobante'][0]['NoCertificado'],
            'certificadoSAT' => isset($map['timbreFiscal']) ? 
                $map['timbreFiscal'][0]['NoCertificadoSAT'] : '',
            'sello' => $map['comprobante'][0]['Sello'],
            'selloSAT' => isset($map['timbreFiscal']) ?
                $map['timbreFiscal'][0]['SelloSAT'] : '',
            'qrcode' => $qrcode,
            'originalString' => InvoiceService::getOriginalString($xml)
        ];
        $businessClient = $invoice->emissor;
        $pdf = PDF::loadView('formats.invoice-format', [
            'invoice' => $invoice,
            'businessClient' => $businessClient,
            'totalString' => $totalString,
            'stamp' => $stamp]);
            
        if (explode('.', PHP_VERSION)[1] >= 4) {
            error_reporting(E_ALL ^ E_DEPRECATED);
        }
        return response($pdf->output())
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="' 
                . $invoice->receiver->name . '.pdf"');
    }

    /**
     * Download requested XML
     */
    public function downloadInvoicePaymentXML(Request $request, $id) {
        $invoice = Invoice::find($id);
        $content = $this->getInvoicePaymentContentXML($invoice);
        return response($content)
            ->header('Content-Type', 'text/xml')
            ->header('Content-Disposition', 'attachment; filename="' . $invoice->receiver->name . '.xml"');
    }

    public function getInvoicePaymentContentXML($invoice) {
        if ($invoice->isStamped()) {
            return Storage::disk('s3')->get($invoice->getStampPath() . $invoice->uuid . '.xml');
        }
        return DocumentServices::buildInvoicePaymentXMLContent($invoice);
    }

    /**
     * Download requested PDF
     */
    public function downloadInvoicePaymentPDF(Request $request, $id) {
        $invoice = Invoice::find($id);
        if (!$invoice->isStamped()) {
            $invoice->code = CodeService::getNext(
                $invoice->emissor->getCodePattern($invoice->business_branch_id),
                $invoice->emissor->getTypeForCodes()
            );
        }

        $qrcode = null;
        if ($invoice->isStamped()) {
            $xml = Storage::disk('s3')->get(
                $invoice->getStampPath() . $invoice->uuid . '.xml');
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => 
                    '/cfdi:Comprobante[][@NoCertificado,@Sello]',
                'timbreFiscal[NoCertificadoSAT,SelloSAT,SelloCFD]' => 
                    '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[][@NoCertificadoSAT,@SelloSAT,@SelloCFD]'
                ], $xml);
            $qr = 'https://verificacfdi.facturaelectronica.sat.gob.mx/default.aspx?id=' 
                . $invoice->uuid . '&re=' . $invoice->emissor->rfc 
                . '&rr=' . $invoice->receiver->rfc 
                . '&tt=' . number_format($invoice->total, 6)
                . '&fe=' . substr($map['timbreFiscal'][0]['SelloCFD'], -8);
            $qrcode = base64_encode(
                FacadesQrCode::format('png')->size(200)->generate($qr)
            );
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]',
                'timbreFiscal[NoCertificadoSAT,SelloSAT]' => '/cfdi:Comprobante/cfdi:Complemento/tfd:TimbreFiscalDigital[][@NoCertificadoSAT,@SelloSAT]'                
            ], $xml);
        } else {
            $xml = DocumentServices::buildInvoiceXMLContent($invoice);
            $map = XmlMapper::mapTo([
                'comprobante[NoCertificado,Sello]' => '/cfdi:Comprobante[][@NoCertificado,@Sello]'
            ], $xml);
        }
        $totalString = strtoupper($this->getStringNumber($invoice->total)) . ' M.N.';
        $stamp = [
            'noCertificado' => $map['comprobante'][0]['NoCertificado'],
            'certificadoSAT' => isset($map['timbreFiscal']) ? 
                $map['timbreFiscal'][0]['NoCertificadoSAT'] : '',
            'sello' => $map['comprobante'][0]['Sello'],
            'selloSAT' => isset($map['timbreFiscal']) ?
                $map['timbreFiscal'][0]['SelloSAT'] : '',
            'qrcode' => $qrcode,
            'originalString' => InvoiceService::getOriginalString($xml)
        ];
        $businessClient = $invoice->emissor;

        if (explode('.', PHP_VERSION)[1] >= 4) {
            error_reporting(E_ALL ^ E_DEPRECATED);
        }
        $pdf = PDF::loadView('formats.invoice-payment-format', [
            'invoice' => $invoice,
            'businessClient' => $businessClient,
            'totalString' => $totalString,
            'stamp' => $stamp]);
        return response($pdf->output())
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="' . $invoice->receiver->name . '.pdf"');
    }

}
