<?php

namespace App\Http\Controllers\ClientBusiness;

use App\Http\Controllers\Controller;
use App\Http\Services\CatCFDIService;
use App\Model\Bank;
use App\Model\BusinessClientEmployee;
use App\Model\Employee;
use App\Model\Partnership;
use App\Model\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class EmployeeController extends Controller
{
    /**
     * List all partnership business
     */
    public function index(Request $request) {      
        $partnership = Auth::user()->partnership();
        
        $page = $request->input('page', 0);
        $search = $request->input('search');
        $sort = 'employees.employee_number';
        $sortOrientation = 'asc';
        $sortTokens = $request->input('sort', 'employees.employee_number:desc');
        if (strpos($sortTokens, ':') != false) {
            list($sort, $sortOrientation) = explode(':', $sortTokens);
        } else {
            $sort = $sortTokens;
        }

        $status = $request->input('status', 'R');

        $query = Employee::where('partnership_id', $partnership->id)
            ->whereNull('canceled_at')
            ->where('status', $status)
            ->orderBy($sort, $sortOrientation);

        if ($search) {
            $query = $query->where(function ($query) use($search) {
                $query->where(DB::raw('concat(name, " ", ' 
                        . 'last_name, " ", last_name2)'), 
                    'like', '%' . $search . '%')
                    ->orWhere('rfc', 'like', '%' . $search . '%')
                    ->orWhere('employee_number', 'like', 
                        '%' . $search . '%');
            });
        }

        $total = $query->count();
        return view('client.employee-index', [
            'data' => $query->skip($page * 25)->take(25)->get(),
            'partnership' => $partnership,
            'status' => $status,
            'total' => $total,
            'page' => $page,
            'search' => $search,
            'sort' => $sort,
            'sortOrientation' => $sortOrientation,
            'pages' => ceil($total / 25)
        ]);
    }

    public function create(Request $request) {
        $partnership = Auth::user()->partnership();
        $employee = new Employee();

        $banks = Bank::all();
        return view('client.new-employee', [
            'partnership' => $partnership,
            'record' => $employee,
            'banks' => $banks,
            'editable' => true,
            'adminFields' => 0,
            'federativeKeys' => CatCFDIService::getFederativeKeys(),
            'payrollTypes' => CatCFDIService::getPayrollTypes(),
            'contractTypes' => CatCFDIService::getContractTypes(),
            'regimeTypes' => CatCFDIService::getRegimeTypes(),
            'workstationRisks' => CatCFDIService::getWorkstationRisks(),
            'paymentPeriods' => CatCFDIService::getPaymentPeriod()
        ]);
    }

    public function edit(Request $request, $id) {
        $partnership = Auth::user()->partnership();
        $employee = Employee::find($id);

        $banks = Bank::all();
        return view('client.new-employee', [
            'partnership' => $partnership,
            'record' => $employee,
            'banks' => $banks,
            'editable' => $employee->status == 'O' || $employee->status == 'F',
            'adminFields' => 0,
            'federativeKeys' => CatCFDIService::getFederativeKeys(),
            'payrollTypes' => CatCFDIService::getPayrollTypes(),
            'contractTypes' => CatCFDIService::getContractTypes(),
            'regimeTypes' => CatCFDIService::getRegimeTypes(),
            'workstationRisks' => CatCFDIService::getWorkstationRisks(),
            'paymentPeriods' => CatCFDIService::getPaymentPeriod()
        ]);
    }

    public function store(Request $request) {
        $id = $request->input('id', 0);

        $partnership = Auth::user()->partnership();
        $query = Employee::whereNull('canceled_at')
            ->where('status', 'R')
            ->where('rfc', strtoupper($request->input('rfc')));
        if ($id != 0) {
            $query->where('id', '<>', $id);
        }
        if ($query->count() > 0) {
            Cache::put('warning', 'El rfc ' . strtoupper($request->input('rfc', '')) 
                . ' ya esta relacionado a otro empleado');
            return redirect()->back()->withInput($request->input());
        }

        $employee = new Employee();
        if ($id != 0) {
            $employee = Employee::find($request['id']);
        }
        $employee->partnership_id = $partnership->id;

        if ($request->input('started-at')) {
            $date = date_create_from_format('Y-m-d', $request->input('started-at'));
            $employee->started_at = $date;
        }
        $employee->employee_number = 0;
        $employee->type = $request->input('type');
        $employee->name = $request->input('name');
        $employee->last_name = $request->input('last-name');
        $employee->last_name2 = $request->input('last-name2');
        $employee->rfc = strtoupper($request->input('rfc'));
        $employee->curp = strtoupper($request->input('curp'));
        $employee->address = $request->input('address');
        $employee->civil_status = $request->input('civil-status');
        $employee->bank_id = $request->input('bank');
        $employee->account = $request->input('account');
        //$employee->card_number = $request->input('card-number');
        $employee->sdi = $request->input('sdi');
        $employee->salary = $request->input('salary');
        $employee->integration_factor = $request->input('integration-factor');
        
        if ($request->input('infonavit-type') == '0') {
            $employee->infonavit_type = null;
            $employee->infonavit_value = null;
        } else {
            $employee->infonavit_type = $request->input('infonavit-type');
            $employee->infonavit_value = $request->input('infonavit-value');
        }

        if ($request->has('nss')) {
            $employee->nss = $request->input('nss');
        }
        if ($request->has('workstation')) {
            $employee->workstation = $request->input('workstation');
        }
        $employee->status = 'O';
        $employee->save();

        if ($request->hasFile('ine-file')) {
            $file = $request->file('ine-file');
            $employee->ine_file = "partnerships/$partnership->code/employees/$employee->id/ine/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->ine_file, $fileContent);
        }
        if ($request->hasFile('address-file')) {
            $file = $request->file('address-file');
            $employee->address_file = "partnerships/$partnership->code/employees/$employee->id/address/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->address_file, $fileContent);
        }
        if ($request->hasFile('rfc-file')) {
            $file = $request->file('rfc-file');
            $employee->rfc_file = "partnerships/$partnership->code/employees/$employee->id/rfc/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->rfc_file, $fileContent);
        }
        if ($request->hasFile('curp-file')) {
            $file = $request->file('curp-file');
            $employee->curp_file = "partnerships/$partnership->code/employees/$employee->id/curp/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->curp_file, $fileContent);
        }
        if ($request->hasFile('infonavit-file')) {
            $file = $request->file('infonavit-file');
            $employee->infonavit_file = "partnerships/$partnership->code/employees/$employee->id/infonavit/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->infonavit_file, $fileContent);
        }
        if ($request->hasFile('nss-file')) {
            $file = $request->file('nss-file');
            $employee->nss_file = "partnerships/$partnership->code/employees/$employee->id/nss/" . $file->getClientOriginalName();
            $employee->save();
            
            $fileContent = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($employee->nss_file, $fileContent);
        }

        foreach ($employee->BusinessClientEmployees as $clientEmployee) {
            $exist = false;
            if ($request->input('business')) {
                for ($i = 0; $i < count($request->input('business')); $i++) {
                    if ($clientEmployee->business_client_id == $request->input('business')[$i]) {
                        $exist = true;
                        break;
                    }
                }
            }
            if (!$exist) {
                $clientEmployee->delete();
            }
        }
        if ($request->input('business')) {
            for ($i = 0; $i < count($request->input('business')); $i++) {
                if (!$employee->isActive($request->input('business')[$i])) {
                    $businessEmployee = new BusinessClientEmployee();
                    $businessEmployee->employee()->associate($employee);
                    $businessEmployee->business_client_id = $request->input('business')[$i];
                    $businessEmployee->save();
                }
            }
        }
        Cache::put('message', 'Se salvo correctamente al empleado ' . $employee->full_name);
        return redirect('/client/employees/' . $employee->id . '/edit');
    }

     /**
     * Stores a invoice of a client
     * @param Request $request Request data
     * @return void 
     */
    public function send(Request $request, $id) {
        $employee = Employee::find($id);
        $employee->status = 'P';
        $employee->save();

        $data = array(
            'user'=> Auth::user(),
            'partnership' => Auth::user()->partnership(),
            'record' => $employee
        );

        if ($employee->type == 'W') {
            $users = User::select('users.email')
                ->join('user_authorizations', 'users.id', '=', 'user_authorizations.user_id')
                ->join('authorizations', 'authorizations.id', '=', 'user_authorizations.authorization_id')
                ->where('authorizations.code', 'WORKER_REQUESTS')
                ->get();
        } else {
            $users = User::select('users.email')
            ->join('user_authorizations', 'users.id', '=', 'user_authorizations.user_id')
            ->join('authorizations', 'authorizations.id', '=', 'user_authorizations.authorization_id')
            ->where('authorizations.code', 'WORKER_REQUESTS')
            ->get();
        }
        /*
        $users = User::select('users.email')
            ->where('id', 1)
            ->get();*/
        Mail::send('emails.employee-request', $data, function($message) use ($users) {
            $message->from('noreply@silifalcon.com', 'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
            $message->subject("Nueva solicitud de alta de empleados");
            foreach ($users as $user) {
                $email = str_replace(" ", "", $user->email);
                if ($email != '' && strpos($email, '@') !== false) {
                    $message->to(str_replace(" ", "", $email));
                }
            }
        });

        Cache::put('message', 'El empleado se envio correctamente a autorizar');
        return redirect("/client/employees/$employee->id/edit");
    }

}
