<?php

namespace App\Http\Controllers\Bills;

use App\Http\Controllers\Controller;
use App\Http\Controllers\StampController;
use App\Http\Services\CertificaCFDI;
use App\Http\Services\CodeService;
use App\Http\Services\DocumentServices;
use App\InvoicePayment;
use App\Model\Invoice;
use App\Model\InvoicePayType;
use App\Model\InvoiceType;
use App\Model\InvoiceUse;
use DateInterval;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;

class InvoicePaymentController extends Controller
{
    
    public function command(Request $request, $id) {
        if ($request->input('action') === 'store') {
            return $this->store($request, $id);
        } else if ($request->input('action') === 'stamp') {
            return $this->stamp($request, $id);
        } else if ($request->input('action') === 'cancel-stamp') {
            return $this->cancelStamp($request);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Shows a form for stamp a invoice
     */
    public function create(Request $request, $id) {
        Session::forget('documents');
        $parent = Invoice::find($id);
        
        $invoice = null;
        $payment = InvoicePayment::join('invoices', 'invoices.id', '=', 'invoice_payments.invoice_id')
            ->whereNull('invoices.uuid')
            ->where('invoice_payments.invoice_related_id', $parent->id)
            ->select('invoice_payments.*')
            ->first();
        $payments = [];
        $invoice = $this->getInvoice($parent, 0);
        $paymentDate = Carbon::now();
        if ($payment) {
            foreach ($invoice->payments as $payment) {
                $paymentDate = $payment->payment_at;
                array_push($payments, $payment);
            }
        } else {
            $payment = $this->getPayment($parent, 0, null);
            array_push($payments, $payment);
        }
        $invoicePayTypes = InvoicePayType::all();

        return view('invoices.invoice-payment', [
            'invoice' => $invoice,
            'parent' => $parent,
            'paymentDate' => $paymentDate,
            'payments' => $payments,
            'parents' => $this->getParents($parent),
            'invoicePayTypes' => $invoicePayTypes
        ]);
    }

    private function validDocuments(Request $request) {
        $countPayments = count($request->input('payment-id'));
        if ($countPayments == 0) {
            Cache::put('warning', 'Por favor agregue por lo menos un documento');
            return redirect()->back()->withInput($request->input());
        }
        $invoices = [];
        for ($i = 0; $i < $countPayments; $i++) {
            $exist = false;
            foreach ($invoices as $invoice) {
                if ($invoice == $request->input('payment-uuid')[$i]) {
                    $exist = true;
                    break;
                }
            }
            if (!$exist) {
                array_push($invoices, $request->input('payment-uuid')[$i]);
            } else {
                $p = Invoice::find($request->input('payment-uuid')[$i]);
                Cache::put('warning', 'El documento: ' . $p->uuid . ' esta repetido, por favor cambielo');
                return redirect()->back()->withInput($request->input());
            }
        }

        for ($i = 0; $i < $countPayments; $i++) {
            if ($request->input('payment-amount')[$i] == 0) {
                Cache::put('warning', 'Los documentos no pueden tener montos de $ 0.0');
                return redirect()->back()->withInput($request->input());
            }
        }
    }

    private function store(Request $request, $id) {
        $this->validDocuments($request);

        $parent = Invoice::find($id);
        $invoiceType = InvoiceType::where('code', 'P')->first();
        $invoiceUse = InvoiceUse::where('code', 'P01')->first();

        $invoice = new Invoice();
        $invoice->user_id = Auth::id();
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        }
        $invoice->invoice_at = Carbon::now();
        $invoice->emissor_id = $parent->emissor->id;
        $invoice->receiver_id = $parent->receiver_id;
        $invoice->business_branch_id = $parent->business_branch_id;
        $invoice->invoice_tax_regime_id = $parent->invoice_tax_regime_id;
        $invoice->invoice_type_id = $invoiceType->id;
        $invoice->invoice_use_id = $invoiceUse->id;
        $invoice->pay_method = 'PUE';
        $invoice->invoice_pay_type_id = $parent->invoice_pay_type_id;
        $date = date_create_from_format('Y-m-d\TH:i', $request->input('invoice-at'));
        $invoice->invoice_at = $date;
        $invoice->total = 0;
        $invoice->subtotal = 0;
        $invoice->invoice_pay_type_id = $request->input('pay-type');
        $invoice->save();

        $ids = [];
        $countPayments = count($request->input('payment-id'));
        for ($i = 0; $i < $countPayments; $i++) {
            $p = Invoice::find($request->input('payment-uuid')[$i]);
            $payment = $this->getPayment($p, $request->input('payment-amount')[$i], $request->input('payment-id')[$i]);
            $payment->invoice_id = $invoice->id;
            
            $seconds = $request->has('payment-at-seconds') ?  $request->input('payment-at-seconds') : 0;
            $date = date_create_from_format('Y-m-d\TH:i', $request->input('payment-at'));
            if ($seconds > 0) {
                $date->add(new DateInterval('PT' . $seconds . 'S'));            
            }
            $payment->payment_at = $date;
            $payment->save();
            array_push($ids, $payment->id);
            $invoice->total += $request->input('payment-amount')[$i];
        }
        $invoice->subtotal = round(($invoice->total / 1.16), 2);        
        $invoice->save();

        InvoicePayment::where('invoice_id', $invoice->id)
            ->whereNotIn('id', $ids)
            ->delete();
        
        return redirect('/tax-documents/invoice-payments/' . $invoice->id . '/edit');
    }

    public function addDocument(Request $request, $id) {
        $parent = Invoice::find($id);
        $payments = [];
        $countPayments = count($request->input('payment-id'));
        $total = 0;
        for ($i = 0; $i < $countPayments; $i++) {
            $p = Invoice::find($request->input('payment-uuid')[$i]);
            $total += $request->input('payment-amount')[$i];
            $payment = $this->getPayment($p, $request->input('payment-amount')[$i], $request->input('payment-id')[$i]);
            array_push($payments, $payment);
        }
        $payment = $this->getPayment($parent, 0, null);
        array_push($payments, $payment);

        $invoicePayTypes = InvoicePayType::all();
        return view('invoices.invoice-payment', [
            'invoice' => $this->getInvoice($parent, $total),
            'parent' => $parent,
            'paymentDate' => $request->input('payment-at', Carbon::now()),
            'payments' => $payments,
            'parents' => $this->getParents($parent),
            'invoicePayTypes' => $invoicePayTypes
        ]);
    }

    public function removeDocument(Request $request, $id) {
        $parent = Invoice::find($request->input('parent-id'));
        $payments = [];
        $countPayments = count($request->input('payment-id'));
        $total = 0;
        for ($i = 0; $i < $countPayments; $i++) {
            if ($i != $id) {
                $p = Invoice::find($request->input('payment-uuid')[$i]);
                $total += $request->input('payment-amount')[$i];
                $payment = $this->getPayment($p, $request->input('payment-amount')[$i], $request->input('payment-id')[$i]);
                array_push($payments, $payment);
            }
        }

        $invoicePayTypes = InvoicePayType::all();
        return view('invoices.invoice-payment', [
            'invoice' => $this->getInvoice($parent, $total),
            'parent' => $parent,
            'paymentDate' => $request->input('payment-at', Carbon::now()),
            'payments' => $payments,
            'parents' => $this->getParents($parent),
            'invoicePayTypes' => $invoicePayTypes
        ]);
    }

    private function getParents($parent) {
        return Invoice::where('receiver_id', $parent->receiver_id)
            ->where('pay_method', 'PPD')
            ->whereNotNull('uuid')
            ->whereNull('canceled_at')
            ->whereRaw("id not in( 
                    select ip.invoice_related_id
                    from invoice_payments ip
                    join invoices p on p.id = ip.invoice_id and p.canceled_at is null
                    where ip.carried_amount = 0 and p.uuid is not null
              )")
            ->get();
    }

    private function getInvoice($parent, $total) {
        $invoice = null;
        $payment = InvoicePayment::join('invoices', 'invoices.id', '=', 'invoice_payments.invoice_id')
            ->whereNull('invoices.uuid')
            ->where('invoice_payments.invoice_related_id', $parent->id)
            ->select('invoice_payments.*')
            ->first();
        if ($payment) {
            $invoice = $payment->invoice;
        } else {
            $invoiceType = InvoiceType::where('code', 'P')->first();
            $invoiceUse = InvoiceUse::where('code', 'P01')->first();

            $invoice = new Invoice();
            $invoice->invoice_at = Carbon::now();
            $invoice->emissor_id = $parent->emissor->id;
            $invoice->receiver_id = $parent->receiver_id;
            $invoice->business_branch_id = $parent->business_branch_id;
            $invoice->invoice_tax_regime_id = $parent->invoice_tax_regime_id;
            $invoice->invoice_type_id = $invoiceType->id;
            $invoice->invoice_use_id = $invoiceUse->id;
            $invoice->pay_method = 'PUE';
            $invoice->pay_type_id = $parent->pay_type_id;
            $invoice->total = $total;
            $invoice->subtotal = round(($invoice->total / 1.16), 2);
        }
        return $invoice;
    }

    private function getPayment($invoice, $amount, $id) {
        $data = $this->getDataPayment($invoice->id);

        $payment = new InvoicePayment();
        if ($id) {
            $payment = InvoicePayment::find($id);
        }
        $payment->invoice_related_id = $invoice->id;
        $payment->pay_method = 'PPD';
        $payment->amount = round($amount, 2);
        $payment->last_amount = round($invoice->total - $data['amount'], 2);
        $payment->carried_amount = round($invoice->total - ($payment->amount + $data['amount']), 2);
        $payment->parciality = $data['parciality'];
        return $payment;
    }

    public function edit(Request $request, $id) {
        $invoice = Invoice::find($id);
        $payments = [];
        $paymentDate = Carbon::now();
        $index = 0;
        foreach ($invoice->payments as $payment) {
            $payment->index = $index ++;
            array_push($payments, $payment);
        }
        $parent = $payment->invoiceRelated;
        $paymentDate = $payment->payment_at;
        $invoicePayTypes = InvoicePayType::all();

        return view('invoices.invoice-payment', [
            'invoice' => $invoice,
            'parent' => $parent,
            'paymentDate' => $paymentDate,
            'payments' => $payments,
            'parents' => $this->getParents($parent),
            'invoicePayTypes' => $invoicePayTypes
        ]);
    }

    public function getDataPayment($id) {
        $payments = InvoicePayment::
            join('invoices', 'invoices.id', '=', 'invoice_payments.invoice_id')
            ->whereNotNull('invoices.uuid')
            ->where('invoice_payments.invoice_related_id', $id)
            ->select('invoice_payments.*')
            ->orderby('invoice_payments.parciality')
            ->get();

        $amount = 0;
        $parciality = 1;
        foreach ($payments as $p) {
            if (!$p->invoice->canceled_at) {
                $amount += $p->amount;
                $parciality ++;
            }
        }
        return [
            'amount' => $amount,
            'parciality' => $parciality
        ];
    }

    public function cancelStamp(Request $request) {
        $invoice = new Invoice();
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        }

        if ($invoice->user_id != Auth::id()) {
            Cache::put('warning', 'El usuario actual no puede cancelar está factura');
            return redirect('/tax-documents/invoice-payments/' . $invoice->id . '/edit');
        }

        set_time_limit(300);   
        if ($invoice->stamp_signer === 'FM') {
            $stamp = new StampController();
            $stamp->cancelInvoice($invoice);
        } else if ($invoice->stamp_signer === 'SI') {
            $xml = DocumentServices::buildInvoiceXMLContent($invoice);
            if ($invoice->isStamped()) {
                $xml = Storage::disk('s3')->get($invoice->getStampPath() . $invoice->uuid . '.xml');
            }
            $stamp = new CertificaCFDI();
            $stamp->cancelInvoice($invoice, $xml);
        }
        return redirect('/tax-documents/invoice-payments/' . $invoice->id . '/edit');
    }
    
    public function stamp(Request $request) {
        $invoice = new Invoice();
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        }
        set_time_limit(300);
        $code = CodeService::getNext(
            $invoice->emissor->getCodePattern($invoice->business_branch_id),
            $invoice->emissor->getTypeForCodes()
        );
        $xmlContent = DocumentServices::buildInvoicePaymentXMLContent($invoice, $code);
        
        //$stamp = new StampController();
        //$stamp->stampInvoice($xmlContent, $invoice);
        CertificaCFDI::stampInvoice($invoice, $xmlContent);        
        if ($invoice->isStamped()) {
            $invoice->code = CodeService::saveNext(
                $invoice->emissor->getCodePattern($invoice->business_branch_id),
                $invoice->emissor->getTypeForCodes()
            );
            $invoice->save();
        }
        return redirect('/tax-documents/invoice-payments/' . $invoice->id . '/edit');
    }

}
