<?php

namespace App\Http\Controllers\Bills;

use App\Http\Controllers\Controller;
use App\Http\Controllers\DownloadController;
use App\Http\Controllers\StampController;
use App\Http\Services\CertificaCFDI;
use App\Http\Services\CodeService;
use App\Http\Services\DocumentServices;
use App\InvoicePayment;
use App\Model\BusinessBranch;
use App\Model\BusinessClient;
use App\Model\BusinessClientConcept;
use App\Model\Invoice;
use App\Model\InvoiceConcept;
use App\Model\InvoicePayType;
use App\Model\InvoiceTaxRegimes;
use App\Model\InvoiceType;
use App\Model\InvoiceUse;
use App\Model\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use File;
use Illuminate\Support\Facades\Auth;
use ZipArchive;

class InvoiceController extends Controller
{
    
    public function command(Request $request) {
        if ($request->input('action') === 'create') {
            return $this->create($request);
        } else if ($request->input('action') === 'store') {
            return $this->store($request);
        } else if ($request->input('action') === 'search') {
            return $this->index($request);
        } else if ($request->input('action') === 'add-concept') {
            return $this->addConcept($request);
        } else if ($request->input('action') === 'stamp') {
            return $this->stamp($request);
        } else if ($request->input('action') === 'destroy') {
            return $this->destroy($request);
        } else if ($request->input('action') === 'cancel-stamp') {
            return $this->cancelStamp($request);
        } else if ($request->input('action') === 'package-invoices') {
            return $this->packageInvoices($request);
        }
        return redirect()->back()->withInput($request->input());
    }

    private function getQuery(Request $request) {
        $from = new Carbon();
        $from->startOfMillennium();
        $to = new Carbon();
        $to->endOfMillennium();

        if ($request->has('from') && !empty($request->get('from'))) {
            $from = new Carbon($request->input('from'));
            $from->startOfDay();
            Session::put('from', $from->format('Y-m-d'));
        } else if (Session::has('from')) {
            $from = new Carbon(Session::get('from'));
            $from->startOfDay();
        }
        if ($request->has('to') && !empty($request->get('to'))) {
            $to = new Carbon($request->input('to'));
            $to->endOfDay();
            Session::put('to', $to->format('Y-m-d'));
        } else if (Session::has('to')) {
            $to = new Carbon(Session::get('to'));
            $to->endOfDay();
        }

        $userId = $request->input('created-by', Auth::id());

        $status = $request->input('status', 'D');
        $payMethod = $request->input('pay-method', 'A');
        $filter = $request->input('filter', null);
        $query = Invoice::join('business_clients', 'business_clients.id', '=', 'invoices.receiver_id')
            ->join('business_clients as emissors', 'emissors.id', '=', 'invoices.emissor_id')
            ->join('invoice_types', 'invoice_types.id', '=', 'invoices.invoice_type_id')
            ->where('invoice_types.code', '<>', 'P');
        if ($userId == 0) {
            $users = User::where('type', 'BI')->get();
            $userIds = [];
            foreach ($users as $user) {
                $userIds[] = $user->id;
            }
            $query->whereNotIn('invoices.user_id', $userIds);
        } else {
            $query->where('invoices.user_id', $userId);
        }
        $query->select('invoices.*');

        if ($status == 'D') {
            $query->whereNotNull('invoices.uuid')
                ->whereNull('invoices.canceled_at');
        } else if ($status == 'M') {
            $query->whereNull('invoices.uuid');
        } else if ($status == 'C') {
            $query->whereNotNull('invoices.canceled_at');
        }
        if ($payMethod == 'PUE' || $payMethod == 'PPD') {
            $query->where('invoices.pay_method', $payMethod);
        }
        $clientName = $request->input('client-name', null);
        if ($clientName) {
            $query->where('business_clients.name', 'like', '%' . $clientName . '%');
        }
        $query->whereBetween('invoices.invoice_at', [$from, $to]);
        if (strlen($filter) > 0) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('emissors.name', 'like', '%' . $filter . '%')
                    ->orWhere('emissors.rfc', 'like', '%' . $filter . '%')
                    ->orWhere('business_clients.rfc', 'like', '%' . $filter . '%')
                    ->orWhere('invoices.code', 'like', '%' . $filter . '%')
                    ->orWhere('invoices.uuid', 'like', '%' . $filter . '%');
            });
        }
        return $query;
    }

    /**
     * Shows a view with the list of invoices
     */
    public function index(Request $request) {
        $page = 0;
        if (!empty($request->input('page'))) {
            $page = $request->input('page');
        }

        $from = new Carbon();
        $from->startOfMillennium();
        $to = new Carbon();
        $to->endOfMillennium();

        if ($request->has('from') && !empty($request->get('from'))) {
            $from = new Carbon($request->input('from'));
            $from->startOfDay();
            Session::put('from', $from->format('Y-m-d'));
        } else if (Session::has('from')) {
            $from = new Carbon(Session::get('from'));
            $from->startOfDay();
        }
        if ($request->has('to') && !empty($request->get('to'))) {
            $to = new Carbon($request->input('to'));
            $to->endOfDay();
            Session::put('to', $to->format('Y-m-d'));
        } else if (Session::has('to')) {
            $to = new Carbon(Session::get('to'));
            $to->endOfDay();
        }

        $status = $request->input('status', 'D');
        $payMethod = $request->input('pay-method', 'A');
        $filter = $request->input('filter', null);
        $clients = BusinessClient::where('type', 'E')
            ->whereNull('canceled_at')
            ->orderBy('name')
            ->get();
        $query = $this->getQuery($request);
        $clientName = $request->input('client-name', null);
        $total = $query->count();
        $query->orderBy('created_at', 'desc')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        $dummyStart = new Carbon();
        $dummyStart->startOfMillennium();
        $dummyEnd = new Carbon();
        $dummyEnd->endOfMillennium();
        return view('invoices.invoice-index', [
            'status' => $status,
            'payMethod' => $payMethod,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'invoices' => $query->get(),
            'total' => $total,
            'clients' => $clients,
            'users' => User::where('type', 'BI')->get(),
            'userId' => $request->input('created-by', Auth::id()),
            'clientName' => $clientName,
            'from' => $from->equalTo($dummyStart) ? null : $from,
            'to' => $to->equalTo($dummyEnd) ? null : $to
        ]);
    }

    /**
     * Shows a form for stamp a invoice
     */
    public function create(Request $request) {
        $receivers = BusinessClient::where('type', 'E')
            ->whereNull('canceled_at')->get(); 
        $emitters = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')->get();
        $branches = BusinessBranch::all();
        
        $invoiceUses = InvoiceUse::all();
        $invoiceTypes = InvoiceType::all();
        $invoiceTaxRegimes = InvoiceTaxRegimes::all();
        $invoicePayTypes = InvoicePayType::all();

        $invoice = new Invoice();
        $invoice->invoice_at = Carbon::now();
        $invoice->invoice_use_id = InvoiceUse::where('code', 'G03')
            ->first()->id;

        return view('invoices.invoice', [
            'invoice' => $invoice,
            'concepts' => $invoice->invoiceConcepts,
            'payments' => [],
            'emitters' => $emitters,
            'branches' => $branches,
            'receivers' => $receivers,
            'invoiceUses' => $invoiceUses,
            'invoiceTypes' => $invoiceTypes,
            'invoiceTaxRegimes' => $invoiceTaxRegimes,
            'invoicePayTypes' => $invoicePayTypes
        ]);
    }

    /**
     * Deletes a list of invoice
     */
    private function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos una factura');
        } else {
            Invoice::whereIn('id', $request->input('selected'))
                ->delete();
            Cache::put('message', 'Las facturas han sido removidas');
        }
        return redirect()->back()->withInput($request->input());
    }

    private function storeInvoice(Request $request) {
        $invoice = new Invoice();
        $invoice->total = 0;
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        } 

        $receiverName = $request->input('receiver-name', '');
        $receiverClient = BusinessClient::where('type', 'E')
                ->where('name', $receiverName)
                ->first();
        if ($receiverClient) {
            $invoice->receiver_id = $receiverClient->id; 
        } else {
            Cache::put('warning', 'Por favor seleccione un cliente');
            return redirect()->back()->withInput($request->input());
        }

        $date = date_create_from_format('Y-m-d\TH:i', $request->input('invoice-at'));
        $invoice->invoice_at = $date;
        
        $invoice->emissor_id = $request->input('client');
        $invoice->business_branch_id = $request->input('branch');
        $invoice->invoice_use_id = $request->input('invoice-use');
        $invoice->invoice_type_id = $request->input('invoice-type');
        $invoice->invoice_tax_regime_id = $request->input('tax-regime');
        $invoice->pay_method = $request->input('pay-method');
        $invoice->invoice_pay_type_id = $request->input('pay-type');        
        $invoice->subtotal = round(($invoice->total / 1.16), 2);
        $invoice->user_id = Auth::id();
        $invoice->save();
        
        $total = 0;
        $subtotal = 0;
        $retention = 0;
        if ($request->input('concept-id')) {
            $countConcepts = count($request->input('concept-id'));
            for ($i = 0; $i < $countConcepts; $i++) {
                $concept = new InvoiceConcept();
                $concept->invoice_id = $invoice->id;
                if ($request->input('concept-id')[$i]) {
                    $concept = $invoice->getInvoiceConcept($request->input('concept-id')[$i]);
                }
                $concept->code = $request->input('concept-code')[$i];
                
                $descriptionAditional = $request->input('concept-description')[$i];
                if ($descriptionAditional) {
                    $descriptionAditional = ltrim($descriptionAditional);
                    if (strlen($descriptionAditional) == 0) {
                        $descriptionAditional = null;
                    }
                }
                $concept->business_client_concept_id = $request->input('concept')[$i];
                $concept->concept = null;

                $concept->description = $descriptionAditional;
                
                $concept->total = round($request->input('concept-total')[$i], 2);
                $concept->subtotal = round($concept->total / 1.16, 2);
                $concept->retention = $invoice->receiver->applyRetention() ?
                    round($concept->subtotal 
                        * ($invoice->receiver->retention / 100), 2) : 0;
                $concept->save();

                $total += $concept->total;
                $subtotal += $concept->subtotal;
                if ($concept->retention != null) {
                    $retention += $concept->retention;
                }
            }
        }
        $invoice->total = round($total, 2);
        $invoice->subtotal = round($subtotal, 2);
        $invoice->retention = round($retention, 2);
        $invoice->save();
        return $invoice;
    }

    /**
     * Saves a invoice header
     */
    public function store(Request $request) {
        $invoice = $this->storeInvoice($request);
        return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');
    }

    /**
     * shows a form for edit a Invoice data
     */
    public function edit(Request $request, $id) {
        $invoice = Invoice::find($id);

        $receivers = BusinessClient::where('type', 'E')->whereNull('canceled_at')->get();
        $emitters = BusinessClient::where('type', 'I')->whereNull('canceled_at')->get();
        $branches = BusinessBranch::all();
        
        $invoiceUses = InvoiceUse::all();
        $invoiceTypes = InvoiceType::all();
        $invoiceTaxRegimes = InvoiceTaxRegimes::all();
        $invoicePayTypes = InvoicePayType::all();
        $businessConcepts = BusinessClientConcept::
            where('business_client_id', $invoice->emissor_id)
            ->get();
        
        $payments = InvoicePayment::where('invoice_related_id', $invoice->id)
            ->orderby('parciality')
            ->get();
        return view('invoices.invoice', [
            'invoice' => $invoice,
            'concepts' => $invoice->invoiceConcepts,
            'payments' => $payments,
            'emitters' => $emitters,
            'branches' => $branches,
            'receivers' => $receivers,
            'invoiceUses' => $invoiceUses,
            'invoiceTypes' => $invoiceTypes,
            'invoiceTaxRegimes' => $invoiceTaxRegimes,
            'invoicePayTypes' => $invoicePayTypes,
            'businessConcepts' => $businessConcepts,
            'startDay' => Carbon::now()->firstOfMonth(),
            'endDay' => Carbon::now()->endOfMonth()
        ]);
    }

    public function addConcept(Request $request) {
        $invoice = $this->storeInvoice($request);

        $concept = new InvoiceConcept();
        $concept->invoice_id = $invoice->id;
        $concept->save();
        return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');
    }

    public function destroyConcept(Request $request, $id) {
        $concept = InvoiceConcept::find($id);
        $concept->delete();
        return redirect('/tax-documents/invoices/' . $concept->invoice_id . '/edit');
    }

    public function destroyPayment(Request $request, $id) {
        $payment = InvoicePayment::where('invoice_id', $id)->first();
        InvoicePayment::where('invoice_id', $id)->delete();
        Invoice::find($id)->delete();
        return redirect('/tax-documents/invoices/' . $payment->invoice_related_id . '/edit');
    }

    public function stamp(Request $request) {
        $invoice = new Invoice();
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        }
        $error = $this->hasErrorsForStamp($invoice);
        if (!!$error) {
            Cache::put('warning', $error);
            return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');    
        }
        set_time_limit(300);
        $code = CodeService::getNext(
            $invoice->emissor->getCodePattern($invoice->business_branch_id),
            $invoice->emissor->getTypeForCodes()
        );
        $xmlContent = DocumentServices::buildInvoiceXMLContent($invoice, $code);
        
        //$stamp = new StampController();
        //$stamp->stampInvoice($xmlContent, $invoice);
        CertificaCFDI::stampInvoice($invoice, $xmlContent);
        if ($invoice->isStamped()) {
            $invoice->code = CodeService::saveNext(
                $invoice->emissor->getCodePattern($invoice->business_branch_id),
                $invoice->emissor->getTypeForCodes()
            );
            $invoice->save();
        }
        return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');
    }

    public function cancelStamp(Request $request) {
        $invoice = new Invoice();
        if ($request['id'] != null) {
            $invoice = Invoice::find($request['id']);
        }

        if ($invoice->user_id != Auth::id()) {
            Cache::put('warning', 'El usuario actual no puede cancelar está factura');
            return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');
        }

        set_time_limit(300);
        $xml = DocumentServices::buildInvoiceXMLContent($invoice);
        if ($invoice->isStamped()) {
            $xml = Storage::disk('s3')->get($invoice->getStampPath() . $invoice->uuid . '.xml');
        }

        if ($invoice->stamp_signer === 'FM') {
            $stamp = new StampController();
            $stamp->cancelInvoice($invoice);    
        } else if ($invoice->stamp_signer === 'SI') {
            $stamp = new CertificaCFDI();
            $stamp->cancelInvoice($invoice, $xml);        
        }
        return redirect('/tax-documents/invoices/' . $invoice->id . '/edit');
    }

    public function hasErrorsForStamp($invoice) {
        $concepts = $invoice->invoiceConcepts;
        if (count($concepts) == 0) {
            return 'Por favor agregue por lo menos un concepto';
        } else if ($invoice->isStamped()) {
            return 'Esta factura ya esta timbrada';
        }
        return false;
    }

    public function packageInvoices(request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos una factura');
            return redirect('/tax-documents/invoices?status=' . $request->input('status'));
        }

        $invoices = Invoice::whereIn('id', $request->input('selected'))->get();
        
        $zipName = 'Timbres.zip';
        $zipDirectory = 'Timbres';
        if (Storage::disk('zips')->exists($zipName)) {
            Storage::disk('zips')->delete($zipName);
        }
        $download = new DownloadController();
        $storagePath  = Storage::disk('zips')->getDriver()
                ->getAdapter()->getPathPrefix();
        $zipper = new ZipArchive();
        if ($zipper->open($storagePath . $zipName, ZipArchive::CREATE) === TRUE) {
            $countInvoices = count($invoices);
            for ($i = 1; $i <= $countInvoices; $i++) {
                set_time_limit(300);
                $r = $invoices[$i - 1];
                $name = $r->code;
                if (!$r->code) {
                    $name = $r->id;
                }
                $data = $download->downloadInvoicePDF($request, $r->id);
                Storage::disk('zips')->put($zipDirectory . '/' . $name . '.pdf', $data);
                $data = $download->downloadInvoiceXML($request, $r->id);
                Storage::disk('zips')->put($zipDirectory . '/' . $name . '.xml', $data);

                $zipper->addFile($storagePath . $zipDirectory . '/' . $name . '.pdf', $name . '.pdf');
                $zipper->addFile($storagePath . $zipDirectory . '/' . $name . '.xml', $name . '.xml');
            }
            $zipper->close();
        }
        if (Storage::disk('zips')->exists($zipDirectory . '')) {
            Storage::disk('zips')->deleteDirectory($zipDirectory . '');
        }
        return response()->download($storagePath . $zipName)->deleteFileAfterSend(true);
    }

}
