<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Services\CertificaCFDI;
use App\Http\Services\CFDIService;
use App\Http\Services\DocumentServices;
use App\Model\BusinessBranch;
use App\Model\BusinessClient;
use App\Model\Employee;
use App\Model\EmployeeScatter;
use App\Model\IsrPeriod;
use App\Model\Partnership;
use App\Model\Payroll;
use App\Model\PayrollReceipt;
use App\Model\ProcessRequest;
use App\Model\ReceiptEmployeeScatter;
use App\Model\Request as ModelRequest;
use App\Model\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class AssimilatedController extends Controller
{

    /**
     * Execute command
     * @param request Controller request
     */
    public function command(Request $request) {
        if ($request->input('action') === 'load-payroll') {
            return $this->loadFromExcel($request);
        } else if ($request->input('action') === 'search') {
            return $this->index($request);
        } else if ($request->input('action') === 'destroy') {
            return $this->destroy($request);
        } else if ($request->input('action') === 'load-preview-employees') {
            return $this->loadPreviewEmployees($request);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Loads data from import code
     */
    public function index(Request $request) {    
        $page = 0;
        if (!empty($request->input('page'))) {
            $page = $request->input('page');
        }

        $importedFrom = new Carbon();
        $importedFrom->startOfMillennium();
        $importedTo = new Carbon();
        $importedTo->endOfMillennium();

        if ($request->has('imported-from')
                && !empty($request->get('imported-from'))) {
            $importedFrom = new Carbon($request->input('imported-from'));
            $importedFrom->startOfDay();
            Session::put('imported-from', $importedFrom->format('Y-m-d'));
        } else if (Session::has('imported-from')) {
            $importedFrom = new Carbon(Session::get('imported-from'));
            $importedFrom->startOfDay();
        }
        if ($request->has('imported-to')
                && !empty($request->get('imported-to'))) {
            $importedTo = new Carbon($request->input('imported-to'));
            $importedTo->endOfDay();
            Session::put('imported-to', $importedTo->format('Y-m-d'));
        } else if (Session::has('imported-to')) {
            $importedTo = new Carbon(Session::get('imported-to'));
            $importedTo->endOfDay();
        }

        $filter = $request->input('filter', null);
        $clients = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')->get();
        $branches = BusinessBranch::all();
        
        $userId = $request->input('created-by', Auth::id());
        $businessId = $request->input('client', null);
        $branchId = $request->input('branch', null);
        $query = Payroll::whereNull('canceled_at')
            ->where('user_id', $userId);
        if ($businessId && $businessId != 'T') {
            $query->where('business_client_id', $businessId);
        }
        if ($branchId && $branchId != 'T') {
            $query->where('business_branch_id', $branchId);
        }
        $query->whereBetween('created_at', [$importedFrom, $importedTo]);
        if (strlen($filter) > 0) {
            $query->where('import_code', 'like', '%' . $filter . '%')
                ->orWhere('name', 'like', '%' . $filter . '%');
        }
        $total = $query->count();
        $query->orderBy('created_at', 'desc')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        $dummyStart = new Carbon();
        $dummyStart->startOfMillennium();
        $dummyEnd = new Carbon();
        $dummyEnd->endOfMillennium();

        return view('payroll.assimilated-index', [
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages,
            'payrolls' => $query->get(),
            'total' => $total,
            'clients' => $clients,
            'users' => User::where('type', 'BI')->get(),
            'userId' => $request->input('created-by', Auth::id()),
            'branches' => $branches,
            'clientId' => $businessId,
            'branchId' => $branchId,
            'importedFrom' => $importedFrom->equalTo($dummyStart) ? null : $importedFrom,
            'importedTo' => $importedTo->equalTo($dummyEnd) ? null : $importedTo
        ]);
    }

    /**
     * Shows a form for load a list of assimilated employees and stamps
     */
    public function create(Request $request) {
        $clients = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')->get();
        $branches = BusinessBranch::all();

        $isrPeriods = IsrPeriod::whereNull('vigented_at')->get();

        return view('payroll.assimilated-create', [
            'clients' => $clients,
            'branches' => $branches,
            'isrPeriods' => $isrPeriods,
            'now' => Carbon::now(),
            'startDay' => Carbon::now()->firstOfMonth(),
            'endDay' => Carbon::now()->endOfMonth()
        ]);
    }

    /**
     * Load data from excel and parse to payroll receipts
     */
    public function loadFromExcel(Request $request) {
        set_time_limit(600);
        if ($request->input('action') && $request->input('action') == 'logout') {
            return redirect('logout');
        }
        $this->validForm();

        $file = $request->file('fel_file');
        $excel = file_get_contents($file->getRealPath());
        $data = base64_encode($excel);
        $map = [
            [ 'column' => 'A', 'target' => 'partnership' ],
            [ 'column' => 'B', 'target' => 'firstName' ],
            [ 'column' => 'C', 'target' => 'lastName'  ],
            [ 'column' => 'D', 'target' => 'name' ],
            [ 'column' => 'E', 'target' => 'curp' ],
            [ 'column' => 'F', 'target' => 'rfc' ],
            [ 'column' => 'G', 'target' => 'total' ],
            [ 'column' => 'H', 'target' => 'request']
        ];

        $mapper = new MapperController();
        $employees = $mapper->getAssimilatedEmployees(
            $mapper->readExcel($data, $map)
        );

        $payroll = $this->getPayroll($request);
        $payrollReceipts = [];

        $rfcs = [];
        $codeRequests = [];
        foreach ($employees as $emp) {
            array_push($rfcs, strtoupper($emp['rfc']));
            if ($emp['request'] && !in_array($emp['request'], $codeRequests)) {
                array_push($codeRequests, strtoupper($emp['request']));
            }
        }
        $modelRequests = [];
        if (count($codeRequests) > 0) {
            $modelRequests = ModelRequest::whereIn('code', $codeRequests)->get();
        }
        $employeeRecords = Employee::whereIn('rfc', $rfcs)->whereNull('canceled_at')->get();
        foreach ($employees as $emp) {
            $currentEmployee = null;
            foreach ($employeeRecords as $empRecord) {
                if ($empRecord->rfc === strtoupper($emp['rfc'])) {
                    $currentEmployee = $empRecord;
                    break;
                }
            }

            if (!isset($payrollReceipts[$emp['rfc']])) {
                $receipt = $this->getPayrollReceipt($request, $payroll, $emp);
                $receipt->num_empleado = $currentEmployee->employee_number;
                $payrollReceipts[$receipt->rfc] = $receipt;
            } else {
                $receipt = $payrollReceipts[$emp['rfc']];
                $receipt->total += round($emp['total'], 2);
                $receipt->isr = CFDIService::getIsrForAssimilatedFromTotal(
                    $receipt->total, $payroll->isr_period_id
                );
                $receipt->subtotal = round($receipt->total + $receipt->isr, 2);
            }
            if ($emp['request']) {
                $requestId = 0;
                foreach ($modelRequests as $modelRequest) {
                    if ($modelRequest->code == $emp['request']) {
                        $requestId = $modelRequest->id;
                    break;
                    }
                }
                $empScatter = EmployeeScatter::where('request_id', $requestId)
                    ->where('employee_id', $currentEmployee->id)
                    ->first();
                if ($empScatter != null) {
                    $receiptEmployeeScatter = new ReceiptEmployeeScatter();
                    $receiptEmployeeScatter->employeeScatter()->associate($empScatter);
                    $receiptEmployeeScatter->payrollReceipt()->associate($receipt);
                    $receiptEmployeeScatter->save();
                }
            }
            $receipt->save();
        }
        return redirect('/payroll/assimilated-employees/' . $payroll->import_code);
    }

    /**
     * Loads data from import code
     */
    public function show(Request $request, $importCode) {
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }
        $partnershipCode = $request->input('partnership', null);
        $partnerships = Partnership::
            join('payroll_receipts', 'payroll_receipts.department', '=', 'partnerships.code')
            ->join('payrolls', 'payrolls.id', '=', 'payroll_receipts.payroll_id')
            ->where('payrolls.import_code', $importCode)
            ->whereNull('payroll_receipts.removed_at')
            ->select('partnerships.*')
            ->distinct()
            ->withTrashed()
            ->get();
        if (!$partnershipCode && count($partnerships) > 0) {
            $partnershipCode = $partnerships[0]->code;
            $request->merge(['partnership' => $partnershipCode]);
        }
        $query = $this->getQueryFilter($request, $importCode)
                ->select('payroll_receipts.*');
        $payroll = Payroll::where('import_code', $importCode)
                    ->get()->first();

        $total = $query->count();
        $query->orderBy('payroll_receipts.department')
            ->orderBy('payroll_receipts.num_empleado')
            ->orderBy('payroll_receipts.name')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        $filter = strtolower( trim($request['filter'] ));

        return view('payroll.assimilated', [
            'page' => $page,
            'pages' => $pages,
            'payroll' => $payroll, 
            'payrollReceipts' => $query->get(),
            'partnershipCode' => $request->input('partnership', null),
            'filter' => $filter,
            'partnerships' => $partnerships,
            'statusId' => $request->input('status', 'A'),
            'totalReceipts' => $total
        ]);
    }

    /**
     * Removes a payroll import
     */
    public function destroy(Request $request) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos una importación');
        } else {
            Payroll::whereIn('id', $request->input('selected'))
                ->update(['canceled_at' => Carbon::now()]);
            Cache::put('message', 'Las importaciones han sido dadas de baja');
        }
        return redirect('/payroll/assimilated-employees');
    }

    /**
     * Execute command
     * @param request Controller request
     */
    public function receiptCommand(Request $request, $importCode) {
        if ($request->input('action') === 'search') {
            $request->merge(['page-index' => 0]);
            return $this->show($request, $importCode);
        } else if ($request->input('action') === 'search-page') {
            return $this->show($request, $importCode);
        } else if ($request->input('action') === 'package-receipts') {
            return $this->packagePayrollReceipts($request, $importCode);
        } else if ($request->input('action') === 'download-package') {
            return $this->downloadPackage($request, $importCode);
        } else if ($request->input('action') === 'remove-receipt') {
            return $this->removePayrollReceipts($request, $importCode);
        } else if ($request->input('action') === 'cancel-stamp') {
            return $this->cancelStamp($request);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Builds a query filter for payroll receipts
     */
    private function getQueryFilter(Request $request, $importCode) {
        $filter = strtolower( trim($request['filter'] ));
        $status = $request->input('status', 'A');
        $partnershipCode = $request->input('partnership', null);
        if (!$partnershipCode) {
            $partnershipCode = PayrollReceipt::
                join('payrolls', 'payrolls.id', '=', 'payroll_receipts.payroll_id')
                ->where('payrolls.import_code', $importCode)
                ->whereNull('payroll_receipts.removed_at')
                ->select('payroll_receipts.department')
                ->distinct()
                ->get()[0]->department;
        }
        $query = PayrollReceipt::
            join('payrolls', 'payrolls.id', '=', 'payroll_receipts.payroll_id')
            ->where('payrolls.import_code', $importCode)
            ->whereNull('payroll_receipts.removed_at');
        if (!$partnershipCode) {
            $partnershipCode = 'T';
        } else if ($partnershipCode != 'T') {
            $query->where('payroll_receipts.department', $partnershipCode);
        }
        if ($status == 'D') {
            $query->whereNotNull('payroll_receipts.uuid')
                ->whereNull('payroll_receipts.canceled_at');
        } else if ($status == 'M') {
            $query->whereNull('payroll_receipts.uuid');
        } else if ($status == 'C') {
            $query->whereNotNull('payroll_receipts.canceled_at');
        }
        if (strlen($filter) > 0) {
            $query->where(
                function($query) use ( $filter ) {
                    $query->where('payroll_receipts.rfc', 'like', 
                        '%' . $filter . '%')
                        ->orWhere(DB::raw('concat(payroll_receipts.name, " ", ' 
                            . 'payroll_receipts.last_name, " ", ' 
                            . 'payroll_receipts.last_name2)'), 
                        'like', '%' . $filter . '%');
                }
            );
        }
        return $query;
    }

    /**
     * Downloads a zip package
     */
    public function downloadPackage(Request $request, $code) {
        $process = ProcessRequest::where('link_code', $code)
                    ->first();
        if ($process) {
            $storagePath  = Storage::disk('zips')->getDriver()
                    ->getAdapter()->getPathPrefix();
            $zipName = $process->code . '.zip';
            if (Storage::disk('zips')->exists($zipName)) {
                return response()->download($storagePath . $zipName);
            }     
        }
        return view('expired');
    }

    /**
     * Remove a payroll receipt
     */
    public function removePayrollReceipts(Request $request, $importCode) {
        if (!$request->input('selected') || empty($request->input('selected'))) {
            Cache::put('warning', 'Debe seleccionar almenos un recibo');
        } else {
            PayrollReceipt::whereIn('id', $request->input('selected'))
                ->whereNull('uuid')
                ->update(['removed_at' => Carbon::now()]);
            Cache::put('message', 'Los recibos que no han sido timbrados, se han removido');
        }
        return redirect('/payroll/assimilated-employees/' . $importCode)
            //->withInput($request->input());
            ->withInput($request->all());
    }

    /**
     * Valid request form
     */
    public function validForm() {
        request() ->validate([
            'pay_date' => 'required',
            'pay_init' => 'required',
            'pay_end' => 'required',
            'invoice_date' => 'required',
            'fel_file' => 'required'
        ], [
            'pay_date.required' => 'La fecha de pago es requerida',
            'pay_init.required' => 'La fecha es requerida',
            'pay_end.required' => 'La fecha es requerida',
            'invoice_date.required' => 'La fecha es requerida',
            'fel_file.required' => 'Seleccione un archivo de excel'
        ]);
    }

    /**
     * Builds a payroll header
     * 
     * @param $request request from assimilated form
     */
    private function getPayroll(Request $request) {
        $client = BusinessClient::find($request['client']);
        $branch = BusinessBranch::find($request['branch']);

        $payroll = new payroll();
        $payroll->import_code = uniqid();
        $payroll->business_client_id = $client->id;
        $payroll->rfc = $client->rfc;
        $payroll->name = $client->name;
        $payroll->business_branch_id = $request['branch'];
        $payroll->postal_code = $branch->postal_code;
        $payroll->pay_period = $request['pay_period'];
        $payroll->isr_period_id = $request['isr_period'];
        $payroll->user_id = Auth::id();
        $payroll->save();

        return $payroll;
    }

    /**
     * Builds a payroll receipt on xml format 
     */
    private function getPayrollReceipt($request, $payroll, $employee) {
        $payrollReceipt = new PayrollReceipt();
        $payrollReceipt->payroll_id = $payroll->id;
        $payrollReceipt->rfc = $employee['rfc'];
        $payrollReceipt->curp = $employee['curp'];
        $payrollReceipt->name = $employee['name'];
        $payrollReceipt->last_name = $employee['firstName'];
        $payrollReceipt->last_name2 = $employee['lastName'];
        $payrollReceipt->department = $employee['partnership'];
        $payrollReceipt->clave_ent_fed = $payroll->businessBranch->prefix;
        $payrollReceipt->total = round($employee['total'], 2);
        $payrollReceipt->isr = CFDIService::getIsrForAssimilatedFromTotal(
            $employee['total'], $payroll->isr_period_id
        );
        $payrollReceipt->pay_period = $request['pay_period'];
        $payrollReceipt->subtotal = round($payrollReceipt->total + 
            $payrollReceipt->isr, 2);
        $payrollReceipt->invoice_date = Carbon::createFromFormat('Y-m-d\TH:i', 
            $request['invoice_date']);
        $payrollReceipt->pay_date = $request['pay_date'];
        $payrollReceipt->pay_init = $request['pay_init'];
        $payrollReceipt->pay_end = $request['pay_end'];
        $payrollReceipt->work_days = $payrollReceipt->pay_init->diff(
            $payrollReceipt->pay_end)->days + 1;
        
        $payrollReceipt->save();
        return $payrollReceipt;
    }

    /**
     * Involves a list of employees to response format
     */
    public function toMapEmployees($employees, $subsideTable) {
        $emps = array();
        foreach ($employees as $emp) {
            array_push($emps, 
                CFDIService::getAssimilatedEmployeeResponse($emp, $subsideTable)
            );
        }
        return $emps;
    }

    /**
     * Shows a form for previsualize a isr of employees
     * @param Request $request 
     * @return Illuminate\View\View|Illuminate\Contracts\View\Factory 
     */
    public function previewAssimilatedEmployees(Request $request) {
        $isrPeriods = IsrPeriod::whereNull('vigented_at')->get();
        return view('payroll.preview-employees', [
            'periods' => $isrPeriods,
            'periodId' => $isrPeriods[0]->id,
            'comission' => 0,
            'amount' => 0,
            'isr' => 0,
            'subtotal' => 0,
            'comissionTotal' => 0,
            'total' => 0
        ]);
    }

    /**
     * Calcs employee isr
     * @param Request $request 
     * @return Illuminate\View\View|Illuminate\Contracts\View\Factory 
     */
    public function loadPreviewEmployees(Request $request) {
        $comission = $request['comission'];
        $amount = round($request['amount'], 2);

        $isr = 0;
        if ($amount > 0) {
            $isr = CFDIService::getIsrForAssimilatedFromTotal(
                $amount, $request['isr-period']
            );
        }
        $subtotal = $amount + $isr;
        $comissionTotal = $comission * $subtotal / 100;
        $total = $subtotal + $comissionTotal;

        $isrPeriods = IsrPeriod::whereNull('vigented_at')->get();
        return view('payroll.preview-employees', [
            'periods' => $isrPeriods,
            'periodId' => $request['isr-period'],
            'comission' => $comission,
            'amount' => $amount,
            'isr' => $isr,
            'subtotal' => $subtotal,
            'comissionTotal' => $comissionTotal,
            'total' => $total
        ]);
    }
}
