<?php

namespace App\Http\Controllers\AdminRequests;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Exports\PaymentScatterExports;
use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\Exports\ScatterExports;
use App\Model\EmployeeScatter;
use App\Model\PayrollWorkerReceipt;
use App\Model\Request as ModelRequest;
use App\Model\ScatterDetail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use ZipArchive;

class RequestController extends Controller
{
    
    /**
     * Execute command
     * @param request Controller request
     */
    public function command(Request $request, $id) {        
        if ($request->input('action') === 'reject') {
            return $this->reject($request, $id);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Retrieves a list of employees
     * @param request Request data
     */
    public function index(Request $request) {
        $query = null;

        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $status = ($request->input('status', 'V'));

        $query = ModelRequest::whereIn('status', ['P', 'F']);
        
        $total = $query->count('id');
        
        $query->orderBy('id')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('requests.scatter-index', [
            'requests' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'page' => $page,
            'pages' => $pages
        ]);
    }

    /**
     * Downloads a zip package
     */
    public function approve(Request $request, $id) {
        $record = ModelRequest::find($id);
        if (Auth::user()->type == 'BI') {
            $record->fac_status = 'A';
        } else if (Auth::user()->type == 'BA') {
            $record->bank_status = 'A';
        }
        if ($record->fac_status && $record->fac_status == 'A' && $record->bank_status && $record->bank_status == 'A') {
            $record->status = 'R';
            EmployeeScatter::select('request_id', $id)->update(['status' => 'R']);
        }
        $record->save();
        $data = array(
            'user'=> Auth::user(),
            'partnership' => Auth::user()->partnership(),
            'record' => $record
        );
        
        if ($record->status == 'R') {
            $employees = EmployeeScatter::select('employee_scatters.*')
                ->join('requests', 'requests.id', '=', 'employee_scatters.request_id')
                ->join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
                ->where('requests.status', 'R')
                ->whereNull('requests.deleted_at')
                //->where('employees.type', 'C')
                ->where('requests.id', $id)
                ->whereNull('employees.canceled_at')
                ->with('scatterDetails', 'employee')
                ->get();
            foreach ($employees as $employee) {
                $employee->closePendingAmount();
            }

            $emails = [
                $record->user->email
            ];
            
            Mail::send('emails.approve-request', $data, function($message) use ($emails) {
                $message->from('noreply@silifalcon.com', 'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
                $message->subject('Solicitud de dispersión de empleados APROBADA');
                foreach ($emails as $email) {
                    $email = str_replace(" ", "", $email);
                    if ($email != '' && strpos($email, '@') !== false) {
                        $message->to(str_replace(" ", "", $email));
                    }
                }          
            });
        }
        return redirect('/requests/scatters');
    }

    /**
     * Downloads a zip package
     */
    public function reject(Request $request, $id) {
        $emails = explode(',', $request->input('email'));
        $count = 0;
        foreach ($emails as $email) {
            $email = str_replace(" ", "", $email);
            if ($email != '' && strpos($email, '@') !== false) {
                $count ++;    
            }
        }
        if ($count == 0) {
            Cache::put('warning', 'Por favor ingrese al menos un correo destinatario');
            return redirect()->back()->withInput($request->input());    
        }

        $record = ModelRequest::find($id);
        if (Auth::user()->type == 'BI') {
            $record->fac_Status = 'F';
            $record->status = 'F';
        } else if (Auth::user()->type == 'BA') {
            $record->bank_Status = 'F';
            $record->status = 'F';
        }
        $record->answer = $request->input('body');
        $record->save();

        if ($record->status == 'F') {
            $data = array(
                'user'=> Auth::user(),
                'partnership' => Auth::user()->partnership(),
                'record' => $record
            );
            Mail::send('emails.approve-request', $data, function($message) use ($emails) {
                $message->from('noreply@silifalcon.com', 
                    'BB DESPACHO INTEGRAL DE SERVICIOS SA DE CV');
                $message->subject('Solicitud de dispersión de empleados RECHAZADA');
                foreach ($emails as $email) {
                    $email = str_replace(" ", "", $email);
                    if ($email != '' && strpos($email, '@') !== false) {
                        $message->to(str_replace(" ", "", $email));
                    }
                }
            });
        }
        return redirect('/requests/scatters');
    }

    /**
     * Downloads a zip package
     */
    public function download(Request $request, $id) {
        set_time_limit(300);
        
        $record = ModelRequest::find($id);
        
        $zipName = 'documentos.zip';
        if (Storage::disk('zips')->exists($zipName)) {
            Storage::disk('zips')->delete($zipName);
        }
        $storagePath  = Storage::disk('zips')->getDriver()->getAdapter()->getPathPrefix();
        $zipper = new ZipArchive();
        if ($zipper->open($storagePath . $zipName, ZipArchive::CREATE) === TRUE) {
            foreach ($record->deposits as $deposit) {
                if ($deposit->voucher_file != '') {
                    Storage::disk('zips')->put('documentos/' . $deposit->getVoucherFile(), 
                    Storage::disk('s3')->get($deposit->voucher_file));
                    $zipper->addFile($storagePath . 'documentos/' . $deposit->getVoucherFile(), $deposit->getVoucherFile());
                }
            }

            $zipper->close();
            if (Storage::disk('zips')->exists('documentos')) {
                Storage::disk('zips')->deleteDirectory('documentos');
            }
        }

        if ($record) {
            $storagePath  = Storage::disk('zips')->getDriver()->getAdapter()->getPathPrefix();
            if (Storage::disk('zips')->exists($zipName)) {
                return response()->download($storagePath . $zipName);
            }
        }
        return redirect()->back()->withInput($request->input());
    }

    public function downloadScatter(Request $request, $id) {
        set_time_limit(300);
        
        if (Auth::user()->type == 'BA') {
            $export = new PaymentScatterExports();
            $export->query = ScatterDetail::select('scatter_details.*')
                ->join('employee_scatters', 'employee_scatters.id', '=', 'scatter_details.employee_scatter_id')
                ->join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
                ->where('employee_scatters.request_id', $id)
                ->whereNotNull('scatter_details.status')
                ->with('employeeScatter', 'requestFile');
    
            return $export->download('dispersion.xlsx');
        } else {
            $export = new ScatterExports();
            $export->query = EmployeeScatter::select('employee_scatters.*')
                ->join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
                ->where('employee_scatters.request_id', $id);
            
            $export->receipts = PayrollWorkerReceipt::
                join('payroll_workers', 'payroll_workers.id', '=', 'payroll_worker_receipts.payroll_worker_id')
                ->where('payroll_workers.request_id', $id)
                ->get();
    
            return $export->download('dispersion.xlsx');
        }        
    }

    /**
     * Shows a employee form for edit his data
     */
    public function show(Request $request, $id) {
        $request = ModelRequest::find($id);
        $employees = $request->employeesWithAll;
        
        $fileSelected = 0;
        if (count($request->requestFilesDeposits) > 0) {
            $fileSelected = $request->requestFilesDeposits[0]->id;
        }

        return view('admin.request', [
            'record' => $request,
            'employeeScatters' => $employees,
            'fileSelected' => $fileSelected,
            'userType' => Auth::user()->type,
            'emailAction' => 'reject'
        ]);
    }

}
