<?php

namespace App\Http\Controllers\AdminRequests;

use App\Http\Controllers\Controller;
use App\Http\Services\CatCFDIService;
use App\Http\Services\CertificaCFDI;
use App\Http\Services\CFDIService;
use App\Http\Services\DocumentServices;
use App\Model\EmployeeScatter;
use App\Model\PayrollWorkerReceipt;
use App\Model\BusinessClient;
use App\Model\BusinessBranch;
use App\Model\IsrPeriod;
use App\Model\PayrollWorker;
use App\Model\Request as ModelRequest;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class PayrollController extends Controller
{
    
    /**
     * Creates payroll workers
     * @param Request $request 
     * @return void 
     */
    public function create(Request $request, $id) {
        $record = ModelRequest::find($id);
        $payroll = new PayrollWorker();
        if (count($record->payrollWorkers) > 0) {
            $payroll = $record->payrollWorkers[0];
        } else {
            $payroll->payed_at = Carbon::now();
            $payroll->invoiced_at = Carbon::now();
            $payroll->pay_init = Carbon::now();
            $payroll->pay_end = Carbon::now();
        }

        $clients = BusinessClient::where('type', 'I')
            ->whereNull('canceled_at')->get();
        $branches = BusinessBranch::all();
        $isrPeriods = IsrPeriod::whereNull('vigented_at')->get();

        return view('requests.payrolls.workers-create', [
            'record' => $record,
            'payroll' => $payroll,
            'isrPeriods' => $isrPeriods,
            'now' => Carbon::now(),
            'clients' => $clients,
            'branches' => $branches,
            'startDay' => Carbon::now()->firstOfMonth(),
            'endDay' => Carbon::now()->endOfMonth()
        ]);
    }

    public function createPayrolls(Request $request, $id) {
        $record = ModelRequest::find($id);
        $payroll = $record->payrollWorkers[0];
        
        $page = 0;
        if (!empty($request->input('page'))) {
            $page = $request->input('page');
        }
        
        $query = PayrollWorkerReceipt::where('payroll_worker_id', $payroll->id);
        $filter = $request->input('filter', '');
        if (strlen($filter) > 0) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('first_name', 'like', '%' . $filter . '%')
                    ->orWhere('middle_name', 'like', '%' . $filter . '%')
                    ->orWhere('last_name', 'like', '%' . $filter . '%')
                    ->orWhere('rfc', 'like', '%' . $filter . '%')
                    ->orWhere('uuid', 'like', '%' . $filter . '%');
            });
        }
        $status = $request->input('status', 'A');
        if ($status == 'D') {
            $query->whereNotNull('uuid')
                ->whereNull('canceled_at');
        } else if ($status == 'M') {
            $query->whereNull('uuid');
        } else if ($status == 'C') {
            $query->whereNotNull('canceled_at');
        }

        $total = $query->count();

        $query->orderBy('first_name')
            ->orderBy('middle_name')
            ->orderBy('last_name')
            ->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        $filter = strtolower( trim($request['filter'] ));

        return view('requests.payrolls.workers', [
            'page' => $page,
            'pages' => $pages,
            'record' => $record,
            'payroll' => $payroll, 
            'payrollReceipts' => $query->get(),
            'filter' => $filter,
            'statusId' => $request->input('status', 'A'),
            'totalReceipts' => $total
        ]);
    }

    public function listPayrolls(Request $request, $code) {
        $payroll = PayrollWorker::where('code', $code)->first();
        
        $page = 0;
        if (!empty($request->input('page'))) {
            $page = $request->input('page');
        }
        
        $query = PayrollWorkerReceipt::where('payroll_worker_id', $payroll->id);
        $filter = $request->input('filter', '');
        if (strlen($filter) > 0) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('first_name', 'like', '%' . $filter . '%')
                    ->orWhere('middle_name', 'like', '%' . $filter . '%')
                    ->orWhere('last_name', 'like', '%' . $filter . '%')
                    ->orWhere('rfc', 'like', '%' . $filter . '%')
                    ->orWhere('uuid', 'like', '%' . $filter . '%');
            });
        }

        $total = $query->count();

        $query->orderBy('first_name')
            ->orderBy('middle_name')
            ->orderBy('last_name')
            ->with('workerReceiptReductions')
            ->offset($page * 25)
            ->limit(25);

        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        $filter = strtolower( trim($request['filter'] ));

        return view('requests.payrolls.workers', [
            'page' => $page,
            'pages' => $pages,
            'record' => $payroll->request,
            'payroll' => $payroll, 
            'payrollReceipts' => $query->get(),
            'filter' => $filter,
            'statusId' => $request->input('status', 'A'),
            'totalReceipts' => $total
        ]);
    }

    /**
     * Loads data from import code
     */
    public function save(Request $request, $id) {
        $record = ModelRequest::find($id);
        $payroll = new PayrollWorker();
        try {
            DB::beginTransaction();  

            $payroll->code = uniqid();
            if ($request->has('id') && $request->input('id') > 0) {
                $payroll = PayrollWorker::find($request->get('id'));
            }
            $payroll->request()->associate($record);
            $payroll->isr_period_id = $request['isr-period'];
            $payroll->salary_min = $request['salary-min'];
            $payroll->umi = $request['umi'];
            $payroll->insurance = $request['insurance'];
            $payroll->payed_at = $request['payed-at'];
            $payroll->pay_init = $request['pay-init'];
            $payroll->pay_end = $request['pay-end'];
            $payroll->invoiced_at = Carbon::createFromFormat('Y-m-d\TH:i', $request['invoiced-at']);
            $payroll->save();

            $payroll->refresh();
            $scatters = EmployeeScatter::select('employee_scatters.*')
                ->join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
                ->where('employee_scatters.request_id', $id)
                ->get();
            foreach ($scatters as $scatter) {
                $receipt = $payroll->getReceipt($scatter->employee_id);
                $reductions = 0;
                if (!$receipt) {
                    $receipt = new PayrollWorkerReceipt();
                    $receipt->payrollWorker()->associate($payroll);
                    $receipt->employeeScatter()->associate($scatter);
                    $receipt->employee_id = $scatter->employee_id;
                } else {
                    $reductions = $receipt->reductions;
                }
                $receipt->isr_period_id = $payroll->isr_period_id;
                $receipt->first_name = $scatter->employee->name;
                $receipt->middle_name = $scatter->employee->last_name;
                $receipt->last_name = $scatter->employee->last_name2;
                $receipt->rfc = $scatter->employee->rfc;
                $receipt->curp = $scatter->employee->curp;
                $receipt->nss = $scatter->employee->nss;

                $receipt->started_at = Carbon::parse($scatter->employee->started_at->format('Y-m-d')) ?? Carbon::parse(Carbon::now()->format('Y-m-d'));
                $receipt->pay_init = Carbon::parse($payroll->pay_init->format('Y-m-d'));
                $receipt->pay_end = Carbon::parse($payroll->pay_end->format('Y-m-d'));

                if ($receipt->started_at > $receipt->pay_init) {
                    $receipt->work_days = $receipt->started_at->diffInDays($receipt->pay_end) + 1;
                } else {
                    $receipt->work_days = $receipt->pay_init->diffInDays($receipt->pay_end) + 1;
                }
                $receipt->payment_period = $scatter->employee->payment_period;
                
                if ($scatter->employee->type == 'W') {
                    $receipt->sdi = $scatter->employee->sdi;
                    $receipt->integration_factor = $scatter->employee->integration_factor;

                    $receipt->payment_period = $scatter->employee->payment_period;
                    $receipt->infonavit_type = $scatter->employee->infonavit_type;
                    $receipt->infonavit_value = $scatter->employee->infonavit_value;

                    $period = IsrPeriod::find($receipt->isr_period_id);
                    if ($period->days == 1) {
                        $subsidy = CFDIService::calcSubsidy($receipt->salary_day, $receipt->isr_period_id) * $receipt->work_days;
                    } else {
                        $subsidy = CFDIService::calcSubsidy($receipt->salary_total, $receipt->isr_period_id);
                    }
                    
                    $receipt->subsidy = round($subsidy > 0 ? $subsidy : 0, 2);
                    $receipt->isr = round($subsidy < 0 ? $subsidy * -1 : 0, 2);

                    $receipt->subsidy_c = CFDIService::calcSubsidyCau($receipt->salary_total, $receipt->salary_day, $receipt->isr_period_id, $receipt->work_days);
                    if ($receipt->infonavit_type) {
                        $receipt->infonavit = round(CFDIService::calcInfonavit($receipt->infonavit_type, $receipt->infonavit_value, 
                            $payroll->umi, $payroll->insurance, $receipt->pay_init, $receipt->work_days, $receipt->sdi), 2);
                    } else {
                        $receipt->infonavit = 0;
                    }
                    $receipt->imss = round(CFDIService::calcIMSS($receipt->sdi, $receipt->work_days, $payroll->umi), 2);
                    $receipt->total = round(($receipt->salary_total + $receipt->subsidy) - ($receipt->imss + $receipt->infonavit + $receipt->isr + $reductions), 2);
                } else {
                    $receipt->sdi = 0;
                    $receipt->integration_factor = 0;
                    
                    $receipt->payment_period = '';
                    $receipt->infonavit_type = '';
                    $receipt->infonavit_value = 0;


                    $receipt->subsidy = 0;
                    $receipt->isr = 0;

                    $receipt->subsidy_c = 0;
                    $receipt->infonavit = 0;
                    $receipt->imss = 0;
                    $receipt->total = 0;
                }
                $receipt->invoice_date = $payroll->invoiced_at;
                $receipt->save();
            }
            DB::commit();
        } catch (Exception $ex) {
            DB::rollBack();
            Log::error($ex);
            return redirect("requests/scatters/$id/show");
        }

        $page = 0;
        $query = PayrollWorkerReceipt::where('payroll_worker_id', $payroll->id);
        $total = $query->count();
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        /*
        return view('requests.payrolls.preview', [
            'page' => $page,
            'pages' => $pages,
            'record' => $payroll,
            'receipts' => $query->get(),
            'filter' => '',
            'totalEmployees' => $total
        ]);*/
        return redirect("/requests/scatters/$record->id/create-payrolls");
    }

    /**
     * Loads data from import code
     */
    public function preview(Request $request, $id) {
        $record = ModelRequest::find($id);
        
        $page = 0;
        if (!empty($request->input('page-index'))) {
            $page = $request->input('page-index');
        }

        $query = EmployeeScatter::join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
            ->where('employee_scatters.request_id', $id);
        $total = $query->count();
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        $filter = strtolower( trim($request['filter'] ));

        $employees = $query->get();
        return view('requests.payrolls.preview', [
            'page' => $page,
            'pages' => $pages,
            'record' => $record,
            'employees' => $employees,
            'filter' => $filter,
            'now' => Carbon::now(),
            'totalEmployees' => $total
        ]);
    }

    public function calcPayroll(Request $request, $id) {
        $record = ModelRequest::find($id);
        
        $page = 0;
        if (!empty($request->input('page'))) {
            $page = $request->input('page');
        }

        $query = EmployeeScatter::join('employees', 'employees.id', '=', 'employee_scatters.employee_id')
            ->where('employee_scatters.request_id', $id);
        $total = $query->count();
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }
        $filter = strtolower( trim($request['filter'] ));

        $query->orderBy('name')
            ->offset($page * 25)
            ->limit(25);

        $employees = $query->get();
        foreach ($employees as $emp) {
            $emp->days = $request->input('day-' . $emp->employee_id);
            if ($emp->employee->type == 'W') {
                $emp->sys = $emp->amount * .7;
                $emp->as = $emp->amount * .3;
            } else {
                $emp->sys = 0;
                $emp->as = $emp->amount;
            }            
        }

        return view('requests.payrolls.preview', [
            'page' => $page,
            'pages' => $pages,
            'record' => $record,
            'employees' => $employees,
            'filter' => $filter,
            'now' => Carbon::now(),
            'totalEmployees' => $total
        ]);
    }

    public function editReceipt(Request $request, $id) {
        $receipt = PayrollWorkerReceipt::find($id);
        $business = $receipt->payrollWorker->businessClient;
        $isrPeriods = IsrPeriod::whereNull('vigented_at')->get();

        return view('requests.payrolls.worker-receipt', [
            'payrollReceipt' => $receipt,
            'isrPeriods' => $isrPeriods,
            'deductionTypes' => CatCFDIService::getDeductionTypes(),
            'business' => $business
        ]);
    }

    public function cancel(Request $request, $id) {        
        set_time_limit(300);
        $receipt = PayrollWorkerReceipt::find($id);
        
        $xml = DocumentServices::buildSysXMLContent($receipt);
        if ($receipt->isStamped()) {
            $xml = Storage::disk('s3')->get($receipt->xml_file);
        }

        $stamp = new CertificaCFDI();
        $stamp->cancelPayrollWorkerReceipt($receipt, $xml);

        return redirect('/requests/scatters/' . $receipt->id . '/receipt');
    }

    public function saveReceipt(Request $request, $id) {
        $receipt = PayrollWorkerReceipt::find($id);
        $payroll = $receipt->payrollWorker;

        $receipt->isr_period_id = $request['isr-period'];
        $receipt->rfc = $request->input('rfc');
        $receipt->curp = $request->input('curp');
        $receipt->first_name = $request->input('name');
        $receipt->middle_name = $request->input('last-name');
        $receipt->last_name = $request->input('last-name2');
        $receipt->work_days = $request->input('work_days');   

        if ($receipt->employee->type == 'W') {
            $period = IsrPeriod::find($receipt->isr_period_id);
            if ($period->days == 1) {
                $subsidy = CFDIService::calcSubsidy($receipt->salary_day, $receipt->isr_period_id) * $receipt->work_days;
            } else {
                $subsidy = CFDIService::calcSubsidy($receipt->salary_total, $receipt->isr_period_id);
            }
            $receipt->subsidy = round($subsidy > 0 ? $subsidy : 0, 2);
            $receipt->isr = round($subsidy < 0 ? $subsidy * -1 : 0, 2);
            $receipt->subsidy_c = round(CFDIService::calcSubsidyCau($receipt->salary_total, $receipt->salary_day, $receipt->isr_period_id, $receipt->work_days), 2);

            if ($receipt->infonavit_type) {
                $receipt->infonavit = round(CFDIService::calcInfonavit($receipt->infonavit_type, $receipt->infonavit_value, 
                    $payroll->umi, $payroll->insurance, $receipt->pay_init, $receipt->work_days, $receipt->sdi), 2);
            } else {
                $receipt->infonavit = 0;
            }
            $receipt->imss = round(CFDIService::calcIMSS($receipt->sdi, $receipt->work_days, $payroll->umi), 2);
            $receipt->total = round(($receipt->salary_total + $receipt->subsidy) - ($receipt->imss + $receipt->infonavit + $receipt->isr + $receipt->reductions), 2);

            WorkerReductionController::saveCurrentDeduction($request, $receipt);
        } else {
            $receipt->subsidy = 0;
            $receipt->isr = 0;

            $receipt->subsidy_c = 0;
            $receipt->infonavit = 0;
            $receipt->imss = 0;
            $receipt->total = 0;
        }
        $receipt->save();

        return redirect("/requests/scatters/$receipt->id/receipt");
    }

    
}
