<?php

namespace App\Http\Controllers\AdminRequests;

use App\Http\Controllers\Controller;
use App\Http\Controllers\DownloadController;
use App\Http\Controllers\Exports\AssimilatedExport;
use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\Exports\ScatterExports;
use App\Model\Employee;
use App\Model\EmployeeScatter;
use App\Model\Invoice;
use App\Model\Partnership;
use App\Model\Request as ModelRequest;
use App\Model\RequestFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use ZipArchive;

class ApprovedRequestController extends Controller
{
    
    /**
     * Retrieves a list of employees
     * @param request Request data
     */
    public function index(Request $request) {
        $page = $request->input('page-index', 0);
        $filter = $request->input('filter', null);
        $partnership = $request->input('partnership', 'T');

        $query = ModelRequest::select('requests.*')
            ->join('business_clients', 'business_clients.id', '=', 'requests.business_client_id')
            ->where('requests.status', 'R');
        if (!empty($filter)) {
            $query = $query->where(function ($query) use($filter) {
                $query->where('business_clients.name', 'like', '%' . $filter . '%');
            });
        }
        
        if ($partnership != 'T') {
            $query->where('business_clients.partnership_id', $request->input('partnership'));
        }
        $total = $query->count('requests.id');
        
        $query->orderBy('requests.id', 'desc')
            ->offset($page * 25)
            ->limit(25);
        $totalPages = floor($total / 25) + ($total % 25 !== 0 ? 1 : 0);
        $pages = [ $page ];
        for ($i = $page - 1; $i >= 0 && $i >= $page - 2; $i--) {
            array_unshift($pages, $i);
        }
        for ($i = $page + 1; $i < $totalPages && count($pages) < 5; $i++) {
            array_push($pages, $i);
        }

        return view('requests.request-index', [
            'requests' => $query->get(),
            'total' => $total,
            'filter' => $filter,
            'userType' => Auth::user()->type,
            'page' => $page,
            'pages' => $pages,
            'partnerships' => Partnership::withTrashed()->orderBy('code')->get(),
            'partnershipId' => $request->input('partnership'),
        ]);
    }

    /**
     * Save a employee
     */
    public function store(Request $request, $id) {
        $code = strtoupper($request->input('invoice-code'));
        $invoice = Invoice::where('code', $code)->first();
        if ($invoice != null) {
            $invoice->request_id = $id;
            $invoice->save();
        }

        return redirect('/requests/scatters/' . $id . '/show');
    }

    public function storeVoucher(Request $request, $id) {
        $record = ModelRequest::find($id);
        $file = $request->file('voucher-file');
        if ($request->has('voucher-file') && $file) {
            $deposit = new RequestFile();
            $deposit->type = 'receipt-pay';
            $deposit->request()->associate($record);

            $deposit->file = $record->businessClient->partnership->code . '/requests/' 
                . $record->id . '/deposits/' . $file->getClientOriginalName();
            $voucher = file_get_contents($file->getRealPath());
            Storage::disk('s3')->put($deposit->file, $voucher);
            $deposit->save();
        }

        return redirect("/requests/scatters/$record->id/show");
    }

    public function removeVoucher(Request $request, $id, $file) {
        $record = ModelRequest::find($id);
        $file = RequestFile::find($file);

        if ($file) {
            Storage::disk('s3')->delete($file->file);
            $file->delete();
        }

        return redirect("/requests/scatters/$record->id/show");
    }

    public function downloadInvoices(Request $request, $id) {
        $record = ModelRequest::find($id);
        $invoices = $record->invoices;
        if (count($invoices) > 0) {
            $zipName = 'Timbres.zip';
            $zipDirectory = 'Timbres';
            if (Storage::disk('zips')->exists($zipName)) {
                Storage::disk('zips')->delete($zipName);
            }
            $download = new DownloadController();
            $storagePath  = Storage::disk('zips')->getDriver()
                    ->getAdapter()->getPathPrefix();
            $zipper = new ZipArchive();
            if ($zipper->open($storagePath . $zipName, ZipArchive::CREATE) === TRUE) {
                $countInvoices = count($invoices);
                for ($i = 1; $i <= $countInvoices; $i++) {
                    set_time_limit(300);
                    $r = $invoices[$i - 1];
                    $name = $r->code;
                    if (!$r->code) {
                        $name = $r->id;
                    }
                    $data = $download->downloadInvoicePDF($request, $r->id);
                    Storage::disk('zips')->put($zipDirectory . '/' . $name . '.pdf', $data);
                    $data = $download->downloadInvoiceXML($request, $r->id);
                    Storage::disk('zips')->put($zipDirectory . '/' . $name . '.xml', $data);

                    $zipper->addFile($storagePath . $zipDirectory . '/' . $name . '.pdf', $name . '.pdf');
                    $zipper->addFile($storagePath . $zipDirectory . '/' . $name . '.xml', $name . '.xml');
                }
                $zipper->close();
            }
            if (Storage::disk('zips')->exists($zipDirectory . '')) {
                Storage::disk('zips')->deleteDirectory($zipDirectory . '');
            }
            return response()->download($storagePath . $zipName)->deleteFileAfterSend(true);
        }
        return redirect()->back()->withInput($request->input());
    }

    /**
     * Save a employee
     */
    public function destroy(Request $request, $id, $invoiceId) {
        $invoice = Invoice::find($invoiceId);
        if ($invoice != null) {
            $invoice->request_id = null;
            $invoice->save();
        }

        return redirect('/requests/scatters/' . $id . '/show');
    }

    public function exportAs(Request $request) {
        set_time_limit(600);
        
        $export = new AssimilatedExport();
        $export->query = EmployeeScatter::select('employee_scatters.*')
            ->join('requests', 'requests.id', '=', 'employee_scatters.request_id')
            ->where('employee_scatters.payed_as', 0)
            ->whereNull('requests.deleted_at')
            ->where('employee_scatters.as', '>', 0)
            ->orderBy('employee_scatters.request_id')
            ->with('request');

        return $export->download('Asimilados.xlsx');
    }

}
