<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\DownloadController;
use App\Model\PayrollReceipt;
use App\Model\ProcessRequest;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class ExportReceiptsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bb:export-receipts { process : process id }
                                { --payroll= : Import code }
                                { --partnership-code= Partnership code }
                                { --status= [D] Stampeds, [M] Missing stampeds}
                                { --send-email : Indicates if will send email }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Export a ZIP file into system';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle() {
        $this->info('start process');

        Log::info('Procesando comando');
        $processId = $this->argument('process');
        $exportCode = $this->option('payroll');
        $partnershipCode = $this->option('partnership-code');
        $status = $this->option('status');
        $sendEmail = $this->option('send-email');

        $process = ProcessRequest::find($processId);
        $process->start();
        $process->save();

        $this->info('Processing: ' . $process->id);
        $this->info('Command: ' . $process->command);

        $query = PayrollReceipt::
            join('payrolls', 'payrolls.id', '=', 'payroll_receipts.payroll_id');
        
        if ($exportCode) {
            $query->where('payrolls.import_code', $exportCode);
        }
        if ($partnershipCode) {
            $query->where('payroll_receipts.department', $partnershipCode);
        }
        if ($status) {
            if ($status == 'D') {
                $query->whereNotNull('payroll_receipts.uuid')
                    ->whereNull('payroll_receipts.canceled_at');
            } else if ($status == 'M') {
                $query->whereNull('uuid');
            } else if ($status == 'C') {
                $query->whereNotNull('payroll_receipts.canceled_at');
            }
        }

        $query->select('payroll_receipts.*');
        $receipts = $query->get();
        $countReceipts = count($receipts);
        $this->info('Found records ' . $countReceipts);
        if ($countReceipts > 0) {
            $zipName = $process->code . '.zip';
            if (Storage::disk('zips')->exists($zipName)) {
                Storage::disk('zips')->delete($zipName);
            }
            $download = new DownloadController();
            $storagePath  = Storage::disk('zips')->getDriver()
                    ->getAdapter()->getPathPrefix();
            $zipper = new ZipArchive();
            if ($zipper->open($storagePath . $zipName, ZipArchive::CREATE) === TRUE) {
                for ($i = 1; $i <= $countReceipts; $i++) {
                    $r = $receipts[$i - 1];
                    $fileName = $process->code . '/' . $i . '.' . $r->getFullName();
                    $data = $download->getContentPDF($r);
                    Storage::disk('zips')->put($fileName . '.pdf', $data);
                    $data = $download->getContentXML($r);
                    Storage::disk('zips')->put($fileName . '.xml', $data);

                    $process->progress = $i * 100 / $countReceipts;
                    $process->save();
                    $zipper->addFile($storagePath . $fileName . '.pdf', $i . '.' . $r->getFullName() . '.pdf');
                    $zipper->addFile($storagePath . $fileName . '.xml', $i . '.' . $r->getFullName() . '.xml');
                }
                $zipper->close();
                if (Storage::disk('zips')->exists($process->code . '')) {
                    Storage::disk('zips')->deleteDirectory($process->code . '');
                }
            }
        }
        $process->finish();
        if ($sendEmail) {
            $process->release();
        }
        $process->save();
    }

}
